//+------------------------------------------------------------------+
//|                                      BrokersDB_PositionSizer.mq5 |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright   "Copyright 2025-2026, BrokersDB.com"
#property link        "https://www.brokersdb.com"
#property version     "1.00"
#property description "BrokersDB Position Sizer — Free position sizing tool"
#property description "Calculates optimal lot size based on risk, entry, and stop-loss."
#property description "Features: 5-tab panel, ATR-based SL/TP, multi-TP, trade execution,"
#property description "trailing stop, breakeven, portfolio risk analysis, margin info."
#property description ""
#property description "Free download at https://www.brokersdb.com"
#property strict

//--- Include all modules
#include "Defines.mqh"
#include "InputParams.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"
#include "Calculator.mqh"
#include "ChartLines.mqh"
#include "Settings.mqh"
#include "TradeManager.mqh"
#include "Panel.mqh"

//+------------------------------------------------------------------+
//| Expert initialization function                                    |
//+------------------------------------------------------------------+
int OnInit()
{
   //--- Terms and Conditions / Disclaimer
   if(!MQLInfoInteger(MQL_TESTER))
   {
      string msg = "WARNING & DISCLAIMER\n\n";
      msg += "By using the BrokersDB Position Sizer, you acknowledge and agree that you are using this software AT YOUR OWN RISK.\n\n";
      msg += "BrokersDB.com and its developers are NOT responsible for any financial losses, damages, or errors that may occur from the use of this tool.\n\n";
      msg += "This is an educational and trading assistance tool, not financial advice.\n\n";
      msg += "Do you agree to these Terms and Conditions?";
      
      if(MessageBox(msg, "BrokersDB Position Sizer - Terms and Conditions", MB_YESNO | MB_ICONWARNING) != IDYES)
      {
         Print("[BrokersDB PS] User declined the Terms and Conditions. Removing EA.");
         return INIT_FAILED;
      }
      
      //--- Check Algo Trading permission
      if(!TerminalInfoInteger(TERMINAL_TRADE_ALLOWED) || !MQLInfoInteger(MQL_TRADE_ALLOWED))
      {
         MessageBox("Algo Trading is currently disabled!\n\nPlease enable 'Allow Algo Trading' in both Terminal Options and EA settings if you wish to automatically execute trades from the panel.", 
                    "BrokersDB Position Sizer - Warning", MB_OK | MB_ICONINFORMATION);
      }
   }

   //--- Initialize global state from input parameters
   InitializeGlobals();

   //--- Try to load saved settings (per-symbol or from file)
   if(InpSettingsFile != "")
   {
      // Load from specified file
      string path = InpSettingsFile;
      if(FileIsExist(path))
      {
         int handle = FileOpen(path, FILE_READ | FILE_TXT);
         if(handle != INVALID_HANDLE) FileClose(handle);
         // Settings file loading delegated to LoadSettings
      }
   }
   else
   {
      LoadSettings(_Symbol);
   }

   //--- Initialize ATR indicator
   if(InpShowATROptions)
   {
      InitATR();
      UpdateATRValue();
   }

   //--- Set up initial entry price
   g_entryPrice = GetCurrentPrice();

   //--- Initialize chart lines
   InitChartLines();

   //--- Perform initial calculation
   g_accountSize = GetAccountSize();
   CalculatePositionSize();

   //--- Create the panel
   CreatePanel();

   //--- Set timer for periodic updates (500ms)
   EventSetMillisecondTimer(500);

   //--- Enable chart events
   ChartSetInteger(0, CHART_EVENT_OBJECT_CREATE, true);
   ChartSetInteger(0, CHART_EVENT_OBJECT_DELETE, true);
   ChartSetInteger(0, CHART_EVENT_MOUSE_MOVE, true);

   Print("[BrokersDB PS] Position Sizer v1.00 initialized on ", _Symbol);
   return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                  |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   //--- Save settings before removing
   SaveSettings(_Symbol);

   //--- Release ATR handle
   ReleaseATR();

   //--- Remove timer
   EventKillTimer();

   //--- Remove panel objects
   DestroyPanel();

   //--- Remove chart lines (unless user wants to keep them)
   if(!InpKeepLinesOnRemove)
      RemoveAllLines();

   Print("[BrokersDB PS] Position Sizer removed. Reason: ", reason);
}

//+------------------------------------------------------------------+
//| Expert tick function                                              |
//+------------------------------------------------------------------+
void OnTick()
{
   //--- Skip heavy processing while dragging panel
   if(g_panelDragging) return;

   //--- Update entry price for instant orders
   if(g_orderMode == ORDER_MODE_INSTANT)
      g_entryPrice = GetCurrentPrice();

   //--- Apply ATR-based SL/TP if set
   if(InpShowATROptions)
   {
      UpdateATRValue();
      ApplyATRStopLoss();
      ApplyATRTakeProfit();
   }

   //--- Recalculate position size
   CalculatePositionSize();

   //--- Update chart lines
   UpdateLinesOnTick();

   //--- Apply trailing stop and breakeven
   ApplyTrailingStop();
   ApplyBreakEven();

   //--- Update panel UI softly
   UpdatePanelData();
}

//+------------------------------------------------------------------+
//| Timer function for panel refresh                                  |
//+------------------------------------------------------------------+
void OnTimer()
{
   //--- Skip all timer processing while dragging panel
   if(g_panelDragging) return;

   //--- Detect symbol change
   if(_Symbol != g_prevSymbol)
   {
      HandleSymbolChange();
      g_prevSymbol = _Symbol;
   }

   //--- Refresh spread display
   if(InpShowSpread != SPREAD_HIDE && !g_panelState.isMinimized)
   {
      string spreadText = "Spread: " + IntegerToString(GetSpreadPoints()) + " pts";
      if(ObjectFind(0, ObjName(MT_LBL_SPREAD)) >= 0)
         ObjectSetString(0, ObjName(MT_LBL_SPREAD), OBJPROP_TEXT, spreadText);
   }

   //--- Refresh panel periodically (live data updates)
   if(g_panelState.isMinimized || g_activeTab == TAB_RISK || g_activeTab == TAB_MARGIN)
   {
      CalculatePositionSize();
      if(g_activeTab == TAB_RISK) CalculatePortfolioRisk();
      UpdatePanelData();
      ChartRedraw(0);
   }
}

//+------------------------------------------------------------------+
//| Chart event handler                                               |
//+------------------------------------------------------------------+
void OnChartEvent(const int id,
                  const long &lparam,
                  const double &dparam,
                  const string &sparam)
{
   //--- Object click events
   if(id == CHARTEVENT_OBJECT_CLICK)
   {
      // Stop dragging if click goes through
      g_panelDragging = false;
      bool handled = HandlePanelClick(sparam);
      if(handled)
      {
         CalculatePositionSize();
         if(g_linesVisible) DrawAllLines();
         RefreshPanel();
         // Unpress button
         ObjectSetInteger(0, sparam, OBJPROP_STATE, false);
         ChartRedraw(0);
      }
      return;
   }

   //--- Edit box end of edit
   if(id == CHARTEVENT_OBJECT_ENDEDIT)
   {
      HandlePanelEdit(sparam);
      return;
   }

   //--- Object drag (line moved)
   if(id == CHARTEVENT_OBJECT_DRAG)
   {
      if(HandleLineDrag(sparam))
      {
         CalculatePositionSize();
         if(g_linesVisible) DrawAllLines();
         RefreshPanel();
      }
      return;
   }

   //--- Mouse Move Event for Panel Dragging
   if(id == CHARTEVENT_MOUSE_MOVE)
   {
      int x = (int)lparam;
      int y = (int)dparam;
      int state = (int)StringToInteger(sparam);

      // Left mouse button pressed
      if((state & 1) == 1)
      {
         int px = g_panelState.posX;
         int py = g_panelState.posY;
         int ph = PANEL_HEADER_H;
         int pw = PANEL_WIDTH;
         
         if(g_panelState.isMinimized) pw = 180; // Minimized width

         // Start dragging if cursor is on the header
         if(!g_panelDragging)
         {
            if(x >= px && x <= px + pw && y >= py && y <= py + ph)
            {
               g_panelDragging = true;
               g_dragOffsetX = x - px;
               g_dragOffsetY = y - py;
            }
         }
         else
         {
            // We are dragging, move entire panel
            int newX = x - g_dragOffsetX;
            int newY = y - g_dragOffsetY;

            if(newX < 0) newX = 0;
            if(newY < 0) newY = 0;
            
            // Move panel via offset shift instead of recreation
            MovePanelTo(newX, newY);
         }
      }
      else
      {
         // Left mouse button released
         g_panelDragging = false;
      }
      return;
   }

   //--- Keyboard events
   if(id == CHARTEVENT_KEYDOWN)
   {
      HandleKeyPress((int)lparam);
      return;
   }
}

//+------------------------------------------------------------------+
//| Handle keyboard shortcuts                                         |
//+------------------------------------------------------------------+
void HandleKeyPress(int keyCode)
{
   bool shift = TerminalInfoInteger(TERMINAL_KEYSTATE_SHIFT) < 0;
   bool ctrl  = TerminalInfoInteger(TERMINAL_KEYSTATE_CONTROL) < 0;

   //--- Parse configured hotkeys
   SHotkey hkTrade    = ParseHotkey(InpHKTrade);
   SHotkey hkOrdType  = ParseHotkey(InpHKOrderType);
   SHotkey hkDir      = ParseHotkey(InpHKDirection);
   SHotkey hkLines    = ParseHotkey(InpHKHideLines);
   SHotkey hkSetSL    = ParseHotkey(InpHKSetSL);
   SHotkey hkSetTP    = ParseHotkey(InpHKSetTP);
   SHotkey hkSetEntry = ParseHotkey(InpHKSetEntry);
   SHotkey hkMinMax   = ParseHotkey(InpHKMinimize);
   SHotkey hkSLMode   = ParseHotkey(InpHKSLMode);
   SHotkey hkTPMode   = ParseHotkey(InpHKTPMode);

   //--- Trade hotkey
   if(keyCode == hkTrade.keyCode && shift == hkTrade.shift && ctrl == hkTrade.ctrl)
   {
      ExecuteTrade();
      return;
   }

   //--- Order type switch
   if(keyCode == hkOrdType.keyCode && shift == hkOrdType.shift && ctrl == hkOrdType.ctrl)
   {
      SwitchOrderMode();
      CalculatePositionSize();
      RefreshPanel();
      if(g_linesVisible) DrawAllLines();
      return;
   }

   //--- Direction switch
   if(keyCode == hkDir.keyCode && shift == hkDir.shift && ctrl == hkDir.ctrl)
   {
      g_tradeDir = (g_tradeDir == TRADE_DIR_LONG) ? TRADE_DIR_SHORT : TRADE_DIR_LONG;
      // Swap SL/TP sides
      double entry = GetEntryPrice();
      if(g_stopLossPrice > 0)
      {
         double slDist = MathAbs(entry - g_stopLossPrice);
         g_stopLossPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry - slDist) : NormalizePrice(entry + slDist);
      }
      if(g_takeProfitPrice > 0)
      {
         double tpDist = MathAbs(g_takeProfitPrice - entry);
         g_takeProfitPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry + tpDist) : NormalizePrice(entry - tpDist);
      }
      CalculatePositionSize();
      RefreshPanel();
      if(g_linesVisible) DrawAllLines();
      return;
   }

   //--- Hide/Show lines
   if(keyCode == hkLines.keyCode && shift == hkLines.shift && ctrl == hkLines.ctrl)
   {
      ToggleLinesVisibility();
      RefreshPanel();
      return;
   }

   //--- Set SL at cursor (mouse position → price)
   if(keyCode == hkSetSL.keyCode && shift == hkSetSL.shift && ctrl == hkSetSL.ctrl)
   {
      int mouseX = 0, mouseY = 0;
      datetime time = 0;
      double price = 0;
      int window = 0;
      ChartXYToTimePrice(0, mouseX, mouseY, window, time, price);
      if(price > 0)
      {
         g_stopLossPrice = NormalizePrice(price);
         CalculatePositionSize();
         RefreshPanel();
         if(g_linesVisible) DrawAllLines();
      }
      return;
   }

   //--- Set TP at cursor
   if(keyCode == hkSetTP.keyCode && shift == hkSetTP.shift && ctrl == hkSetTP.ctrl)
   {
      int mouseX = 0, mouseY = 0;
      datetime time = 0;
      double price = 0;
      int window = 0;
      ChartXYToTimePrice(0, mouseX, mouseY, window, time, price);
      if(price > 0)
      {
         g_takeProfitPrice = NormalizePrice(price);
         if(g_tpCount > 0) g_tpLevels[0].price = g_takeProfitPrice;
         CalculatePositionSize();
         RefreshPanel();
         if(g_linesVisible) DrawAllLines();
      }
      return;
   }

   //--- Set Entry at cursor
   if(keyCode == hkSetEntry.keyCode && shift == hkSetEntry.shift && ctrl == hkSetEntry.ctrl)
   {
      if(g_orderMode == ORDER_MODE_INSTANT)
         g_orderMode = ORDER_MODE_PENDING;
      int mouseX = 0, mouseY = 0;
      datetime time = 0;
      double price = 0;
      int window = 0;
      ChartXYToTimePrice(0, mouseX, mouseY, window, time, price);
      if(price > 0)
      {
         g_entryPrice = NormalizePrice(price);
         CalculatePositionSize();
         RefreshPanel();
         if(g_linesVisible) DrawAllLines();
      }
      return;
   }

   //--- Minimize/Maximize
   if(keyCode == hkMinMax.keyCode && shift == hkMinMax.shift && ctrl == hkMinMax.ctrl)
   {
      g_panelState.isMinimized = !g_panelState.isMinimized;
      DestroyPanel();
      CreatePanel();
      return;
   }

   //--- SL mode switch (points/level)
   if(keyCode == hkSLMode.keyCode && shift == hkSLMode.shift && ctrl == hkSLMode.ctrl)
   {
      g_slInPoints = !g_slInPoints;
      if(g_slInPoints)
         g_slPointsValue = CalcSLDistancePoints();
      RefreshPanel();
      return;
   }

   //--- TP mode switch (points/level)
   if(keyCode == hkTPMode.keyCode && shift == hkTPMode.shift && ctrl == hkTPMode.ctrl)
   {
      g_tpInPoints = !g_tpInPoints;
      if(g_tpInPoints)
         g_tpPointsValue = CalcTPDistancePoints();
      RefreshPanel();
      return;
   }
}

//+------------------------------------------------------------------+
//| Handle symbol change on chart                                     |
//+------------------------------------------------------------------+
void HandleSymbolChange()
{
   //--- Save current symbol settings
   SaveSettings(g_prevSymbol);

   switch(InpSymbolChange)
   {
      case SYMBOL_CHANGE_EACH_OWN:
         //--- Try loading settings for new symbol
         if(!LoadSettings(_Symbol))
         {
            //--- No saved settings, reset to defaults
            InitializeGlobals();
         }
         break;

      case SYMBOL_CHANGE_HARD_RESET:
         InitializeGlobals();
         break;

      case SYMBOL_CHANGE_KEEP:
         //--- Keep panel settings as-is
         break;
   }

   //--- Update trade symbol
   g_tradeSymbol = (InpTradeSymbol == "") ? _Symbol : InpTradeSymbol;

   //--- Reset entry price to current
   g_entryPrice = GetCurrentPrice();

   //--- Reinitialize ATR if needed
   if(InpShowATROptions)
   {
      ReleaseATR();
      InitATR();
      UpdateATRValue();
   }

   //--- Recalculate
   g_accountSize = GetAccountSize();
   CalculatePositionSize();

   //--- Redraw
   if(g_linesVisible)
   {
      RemoveAllLines();
      InitChartLines();
   }
   DestroyPanel();
   CreatePanel();
}
//+------------------------------------------------------------------+
