//+------------------------------------------------------------------+
//|                                                   Calculator.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef CALCULATOR_MQH
#define CALCULATOR_MQH

#include "Defines.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"

//+------------------------------------------------------------------+
//| Position Size Calculation Engine                                   |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Initialize ATR indicator handle                                   |
//+------------------------------------------------------------------+
bool InitATR()
{
   if(g_atrHandle != INVALID_HANDLE)
      IndicatorRelease(g_atrHandle);

   ENUM_TIMEFRAMES tf = (g_atrTimeframe == PERIOD_CURRENT) ? _Period : g_atrTimeframe;
   g_atrHandle = iATR(_Symbol, tf, g_atrPeriod);

   if(g_atrHandle == INVALID_HANDLE)
   {
      Print("[BrokersDB PS] Failed to create ATR indicator handle");
      return false;
   }
   return true;
}

//+------------------------------------------------------------------+
//| Release ATR indicator handle                                      |
//+------------------------------------------------------------------+
void ReleaseATR()
{
   if(g_atrHandle != INVALID_HANDLE)
   {
      IndicatorRelease(g_atrHandle);
      g_atrHandle = INVALID_HANDLE;
   }
}

//+------------------------------------------------------------------+
//| Update ATR value from indicator                                   |
//+------------------------------------------------------------------+
void UpdateATRValue()
{
   if(g_atrHandle == INVALID_HANDLE)
   {
      if(!InitATR()) return;
   }

   double atrBuffer[1];
   int shift = (InpATRCandle == ATR_CANDLE_PREVIOUS) ? 1 : 0;

   if(CopyBuffer(g_atrHandle, 0, shift, 1, atrBuffer) > 0)
      g_atrValue = atrBuffer[0];
   else
      g_atrValue = 0;
}

//+------------------------------------------------------------------+
//| Apply ATR-based SL if ATR multiplier > 0                          |
//+------------------------------------------------------------------+
void ApplyATRStopLoss()
{
   if(g_atrMultiplierSL <= 0 || g_atrValue <= 0) return;

   double slDistance = g_atrValue * g_atrMultiplierSL;
   double entry = GetEntryPrice();
   int spread = GetSpreadPoints();
   double spreadPrice = PointsToPrice(spread);

   if(g_tradeDir == TRADE_DIR_LONG)
   {
      g_stopLossPrice = NormalizePrice(entry - slDistance);
      if(g_spreadAdjSL)
         g_stopLossPrice = NormalizePrice(g_stopLossPrice - spreadPrice);
   }
   else
   {
      g_stopLossPrice = NormalizePrice(entry + slDistance);
      if(g_spreadAdjSL)
         g_stopLossPrice = NormalizePrice(g_stopLossPrice + spreadPrice);
   }
}

//+------------------------------------------------------------------+
//| Apply ATR-based TP if ATR multiplier > 0                          |
//+------------------------------------------------------------------+
void ApplyATRTakeProfit()
{
   if(g_atrMultiplierTP <= 0 || g_atrValue <= 0) return;

   double tpDistance = g_atrValue * g_atrMultiplierTP;
   double entry = GetEntryPrice();
   int spread = GetSpreadPoints();
   double spreadPrice = PointsToPrice(spread);

   if(g_tradeDir == TRADE_DIR_LONG)
   {
      g_takeProfitPrice = NormalizePrice(entry + tpDistance);
      if(g_spreadAdjTP)
         g_takeProfitPrice = NormalizePrice(g_takeProfitPrice - spreadPrice);
   }
   else
   {
      g_takeProfitPrice = NormalizePrice(entry - tpDistance);
      if(g_spreadAdjTP)
         g_takeProfitPrice = NormalizePrice(g_takeProfitPrice + spreadPrice);
   }

   //--- Update first TP level
   if(g_tpCount > 0)
      g_tpLevels[0].price = g_takeProfitPrice;
}

//+------------------------------------------------------------------+
//| Calculate SL distance in points                                    |
//+------------------------------------------------------------------+
int CalcSLDistancePoints()
{
   double entry = GetEntryPrice();
   if(entry == 0 || g_stopLossPrice == 0) return 0;

   double distance = MathAbs(entry - g_stopLossPrice);
   return PriceToPoints(distance);
}

//+------------------------------------------------------------------+
//| Calculate TP distance in points                                    |
//+------------------------------------------------------------------+
int CalcTPDistancePoints()
{
   double entry = GetEntryPrice();
   if(entry == 0 || g_takeProfitPrice == 0) return 0;

   double distance = MathAbs(g_takeProfitPrice - entry);
   return PriceToPoints(distance);
}

//+------------------------------------------------------------------+
//| Calculate commission cost for a position                           |
//+------------------------------------------------------------------+
double CalcCommissionCost(double lots)
{
   if(g_commission == 0) return 0;

   if(g_commType == COMMISSION_CURRENCY)
   {
      // Commission is in account currency per lot (one-way), multiply by 2 for round trip
      return g_commission * lots * 2.0;
   }
   else
   {
      // Commission is percentage of trade value
      string symbol = GetTradeSymbol();
      double contractSize = SymbolInfoDouble(symbol, SYMBOL_TRADE_CONTRACT_SIZE);
      double price = GetEntryPrice();
      double tradeValue = lots * contractSize * price;

      // Need to convert to account currency if necessary
      // For simplicity, use tick value approach
      double tickValue = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
      double tickSize = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE);
      if(tickSize > 0)
         tradeValue = lots * contractSize * tickValue * price / tickSize;

      return tradeValue * g_commission / 100.0 * 2.0;
   }
}

//+------------------------------------------------------------------+
//| Core lot size calculation                                          |
//+------------------------------------------------------------------+
void CalculatePositionSize()
{
   string symbol  = GetTradeSymbol();
   double entry   = GetEntryPrice();
   double sl      = g_stopLossPrice;
   double tp      = g_takeProfitPrice;

   //--- Get account size
   g_accountSize = GetAccountSize();

   //--- SL distance
   if(entry == 0 || sl == 0)
   {
      ZeroCalcResult();
      return;
   }

   double slDistance = MathAbs(entry - sl);
   int slPoints = PriceToPoints(slDistance, symbol);
   if(slPoints <= 0)
   {
      ZeroCalcResult();
      return;
   }

   g_calcResult.slDistancePoints = slPoints;

   //--- TP distance
   if(tp != 0)
   {
      double tpDistance = MathAbs(tp - entry);
      g_calcResult.tpDistancePoints = PriceToPoints(tpDistance, symbol);
   }
   else
   {
      g_calcResult.tpDistancePoints = 0;
   }

   //--- Point value per lot
   double pointValuePerLot = GetPointValue(symbol);
   if(pointValuePerLot == 0)
   {
      ZeroCalcResult();
      return;
   }

   //--- Calculate risk money
   double riskMoney;
   if(g_riskMoney > 0)
   {
      riskMoney = g_riskMoney;
      g_riskPercent = (g_accountSize > 0) ? (riskMoney / g_accountSize * 100.0) : 0;
   }
   else
   {
      riskMoney = g_accountSize * g_riskPercent / 100.0;
      g_riskMoney = 0; // Keep it as 0 to indicate we're using percentage
   }

   //--- Calculate raw lot size: risk / (SL_points * point_value_per_lot)
   double riskPerLot = slPoints * pointValuePerLot;

   //--- Account for commission
   double commPerLot = 0;
   if(g_commission > 0)
   {
      commPerLot = CalcCommissionCost(1.0);
   }

   double totalRiskPerLot = riskPerLot + commPerLot;
   double rawLots = 0;
   if(totalRiskPerLot > 0)
      rawLots = riskMoney / totalRiskPerLot;

   //--- Store unadjusted result
   g_calcResult.lotSize = rawLots;

   //--- Adjusted position size (broker limits)
   g_calcResult.lotSizeAdjusted = NormalizeLots(rawLots, symbol);

   //--- Cap by margin if requested
   if(InpCapPSByMargin)
   {
      double maxByMargin = CalcMaxLotsByMargin();
      if(g_calcResult.lotSizeAdjusted > maxByMargin && maxByMargin > 0)
         g_calcResult.lotSizeAdjusted = NormalizeLots(maxByMargin, symbol);
   }

   //--- Risk from input lot size
   g_calcResult.riskMoney = rawLots * totalRiskPerLot;
   g_calcResult.riskPercent = (g_accountSize > 0) ? (g_calcResult.riskMoney / g_accountSize * 100.0) : 0;

   //--- Risk from adjusted lot size (result)
   g_calcResult.riskMoneyResult = g_calcResult.lotSizeAdjusted * totalRiskPerLot;
   g_calcResult.riskPercentResult = (g_accountSize > 0) ? (g_calcResult.riskMoneyResult / g_accountSize * 100.0) : 0;

   //--- Reward calculations
   if(g_calcResult.tpDistancePoints > 0)
   {
      double rewardPerLot = g_calcResult.tpDistancePoints * pointValuePerLot;
      g_calcResult.rewardMoney = rawLots * (rewardPerLot - commPerLot);
      g_calcResult.rewardMoneyResult = g_calcResult.lotSizeAdjusted * (rewardPerLot - commPerLot);

      //--- R:R ratio
      if(g_calcResult.riskMoney > 0)
         g_calcResult.rewardRiskRatio = g_calcResult.rewardMoney / g_calcResult.riskMoney;
      else
         g_calcResult.rewardRiskRatio = 0;

      if(g_calcResult.riskMoneyResult > 0)
         g_calcResult.rewardRiskResult = g_calcResult.rewardMoneyResult / g_calcResult.riskMoneyResult;
      else
         g_calcResult.rewardRiskResult = 0;
   }
   else
   {
      g_calcResult.rewardMoney = 0;
      g_calcResult.rewardMoneyResult = 0;
      g_calcResult.rewardRiskRatio = 0;
      g_calcResult.rewardRiskResult = 0;
   }

   //--- Point value for calculated position
   g_calcResult.pointValue = g_calcResult.lotSizeAdjusted * pointValuePerLot;

   //--- Margin calculations
   CalculateMargin();

   //--- Distribute among multi-TPs
   DistributeMultiTP();
}

//+------------------------------------------------------------------+
//| Calculate margin-related values                                    |
//+------------------------------------------------------------------+
void CalculateMargin()
{
   string symbol = GetTradeSymbol();
   double lots = g_calcResult.lotSizeAdjusted;
   double entry = GetEntryPrice();
   ENUM_ORDER_TYPE orderType = (g_tradeDir == TRADE_DIR_LONG) ? ORDER_TYPE_BUY : ORDER_TYPE_SELL;

   //--- Get position margin
   double margin = 0;
   if(!OrderCalcMargin(orderType, symbol, lots, entry, margin))
      margin = 0;

   //--- Apply custom leverage if set
   if(g_customLeverage > 0)
   {
      long defaultLeverage = AccountInfoInteger(ACCOUNT_LEVERAGE);
      if(defaultLeverage > 0 && margin > 0)
         margin = margin * defaultLeverage / g_customLeverage;
   }

   g_calcResult.positionMargin = margin;
   g_calcResult.futureUsedMargin = AccountInfoDouble(ACCOUNT_MARGIN) + margin;
   g_calcResult.futureFreeMargin = AccountInfoDouble(ACCOUNT_EQUITY) - g_calcResult.futureUsedMargin;
}

//+------------------------------------------------------------------+
//| Calculate maximum position size by available margin                |
//+------------------------------------------------------------------+
double CalcMaxLotsByMargin()
{
   string symbol = GetTradeSymbol();
   double entry = GetEntryPrice();
   if(entry == 0) return 0;

   ENUM_ORDER_TYPE orderType = (g_tradeDir == TRADE_DIR_LONG) ? ORDER_TYPE_BUY : ORDER_TYPE_SELL;

   //--- Get margin per 1 lot
   double marginPerLot = 0;
   if(!OrderCalcMargin(orderType, symbol, 1.0, entry, marginPerLot) || marginPerLot <= 0)
      return 0;

   //--- Apply custom leverage
   if(g_customLeverage > 0)
   {
      long defaultLeverage = AccountInfoInteger(ACCOUNT_LEVERAGE);
      if(defaultLeverage > 0)
         marginPerLot = marginPerLot * defaultLeverage / g_customLeverage;
   }

   //--- Available free margin
   double freeMargin = AccountInfoDouble(ACCOUNT_MARGIN_FREE);
   if(freeMargin <= 0) return 0;

   double maxLots = freeMargin / marginPerLot;
   g_calcResult.maxLotsByMargin = NormalizeLots(maxLots, symbol);
   return g_calcResult.maxLotsByMargin;
}

//+------------------------------------------------------------------+
//| Distribute position size among multi-TP levels                     |
//+------------------------------------------------------------------+
void DistributeMultiTP()
{
   if(g_tpCount <= 1) return;

   double totalLots = g_calcResult.lotSizeAdjusted;

   for(int i = 0; i < g_tpCount; i++)
   {
      double share = g_tpLevels[i].sharePct / 100.0;
      g_tpLevels[i].lots = NormalizeLots(totalLots * share);
   }
}

//+------------------------------------------------------------------+
//| Calculate risk from a given position size (reverse calc)           |
//+------------------------------------------------------------------+
void CalcRiskFromPositionSize(double lots)
{
   string symbol = GetTradeSymbol();
   double entry = GetEntryPrice();
   double sl = g_stopLossPrice;

   if(entry == 0 || sl == 0 || lots <= 0) return;

   int slPoints = PriceToPoints(MathAbs(entry - sl), symbol);
   double pointValuePerLot = GetPointValue(symbol);
   double commPerLot = CalcCommissionCost(1.0);
   double totalRiskPerLot = slPoints * pointValuePerLot + commPerLot;

   double riskMoney = lots * totalRiskPerLot;
   g_riskMoney = riskMoney;
   g_riskPercent = (g_accountSize > 0) ? (riskMoney / g_accountSize * 100.0) : 0;

   g_calcResult.lotSize = lots;
   g_calcResult.lotSizeAdjusted = lots;
   g_calcResult.riskMoney = riskMoney;
   g_calcResult.riskPercent = g_riskPercent;
   g_calcResult.riskMoneyResult = riskMoney;
   g_calcResult.riskPercentResult = g_riskPercent;
}

//+------------------------------------------------------------------+
//| Zero out calculation results                                       |
//+------------------------------------------------------------------+
void ZeroCalcResult()
{
   ZeroMemory(g_calcResult);
}

//+------------------------------------------------------------------+
//| Calculate portfolio risk (for Risk tab)                            |
//+------------------------------------------------------------------+
void CalculatePortfolioRisk()
{
   ZeroMemory(g_portfolioRisk);

   double accountSize = GetAccountSize();
   string currentSymbol = _Symbol;

   double totalRiskMoney = 0;
   double totalRewardMoney = 0;
   double totalLots = 0;

   //--- Scan open positions
   if(g_includeOrders != INCLUDE_ORDERS_PENDING)
   {
      int total = PositionsTotal();
      for(int i = 0; i < total; i++)
      {
         ulong ticket = PositionGetTicket(i);
         if(ticket == 0) continue;
         if(!PositionSelectByTicket(ticket)) continue;

         string sym = PositionGetString(POSITION_SYMBOL);

         //--- Symbol filter
         if(g_includeSymbols == INCLUDE_SYMBOLS_CURRENT && sym != currentSymbol) continue;
         if(g_includeSymbols == INCLUDE_SYMBOLS_OTHER && sym == currentSymbol) continue;

         //--- Direction filter
         ENUM_POSITION_TYPE posType = (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE);
         if(g_includeDirections == INCLUDE_DIRECTIONS_BUY && posType != POSITION_TYPE_BUY) continue;
         if(g_includeDirections == INCLUDE_DIRECTIONS_SELL && posType != POSITION_TYPE_SELL) continue;

         double posLots    = PositionGetDouble(POSITION_VOLUME);
         double posSL      = PositionGetDouble(POSITION_SL);
         double posTP      = PositionGetDouble(POSITION_TP);
         double posOpen    = PositionGetDouble(POSITION_PRICE_OPEN);

         totalLots += posLots;

         //--- Risk from SL
         if(posSL != 0)
         {
            double slDist = MathAbs(posOpen - posSL);
            int slPts = PriceToPoints(slDist, sym);
            double pv = GetPointValue(sym);
            totalRiskMoney += slPts * pv * posLots;
         }
         else if(!g_ignoreNoSL)
         {
            // No SL: risk is considered as current unrealized loss (worst case)
            double profit = PositionGetDouble(POSITION_PROFIT);
            if(profit < 0)
               totalRiskMoney += MathAbs(profit);
         }

         //--- Reward from TP
         if(posTP != 0)
         {
            double tpDist = MathAbs(posTP - posOpen);
            int tpPts = PriceToPoints(tpDist, sym);
            double pv = GetPointValue(sym);
            totalRewardMoney += tpPts * pv * posLots;
         }
      }
   }

   //--- Scan pending orders
   if(g_includeOrders != INCLUDE_ORDERS_OPEN)
   {
      int total = OrdersTotal();
      for(int i = 0; i < total; i++)
      {
         ulong ticket = OrderGetTicket(i);
         if(ticket == 0) continue;

         string sym = OrderGetString(ORDER_SYMBOL);

         //--- Symbol filter
         if(g_includeSymbols == INCLUDE_SYMBOLS_CURRENT && sym != currentSymbol) continue;
         if(g_includeSymbols == INCLUDE_SYMBOLS_OTHER && sym == currentSymbol) continue;

         //--- Direction filter
         ENUM_ORDER_TYPE orderType = (ENUM_ORDER_TYPE)OrderGetInteger(ORDER_TYPE);
         bool isBuy = (orderType == ORDER_TYPE_BUY_LIMIT || orderType == ORDER_TYPE_BUY_STOP || orderType == ORDER_TYPE_BUY_STOP_LIMIT);
         bool isSell = !isBuy;
         if(g_includeDirections == INCLUDE_DIRECTIONS_BUY && !isBuy) continue;
         if(g_includeDirections == INCLUDE_DIRECTIONS_SELL && !isSell) continue;

         double ordLots = OrderGetDouble(ORDER_VOLUME_CURRENT);
         double ordSL   = OrderGetDouble(ORDER_SL);
         double ordTP   = OrderGetDouble(ORDER_TP);
         double ordOpen = OrderGetDouble(ORDER_PRICE_OPEN);

         totalLots += ordLots;

         if(ordSL != 0)
         {
            double slDist = MathAbs(ordOpen - ordSL);
            int slPts = PriceToPoints(slDist, sym);
            double pv = GetPointValue(sym);
            totalRiskMoney += slPts * pv * ordLots;
         }

         if(ordTP != 0)
         {
            double tpDist = MathAbs(ordTP - ordOpen);
            int tpPts = PriceToPoints(tpDist, sym);
            double pv = GetPointValue(sym);
            totalRewardMoney += tpPts * pv * ordLots;
         }
      }
   }

   //--- Current portfolio
   g_portfolioRisk.currentRiskMoney    = totalRiskMoney;
   g_portfolioRisk.currentRiskPercent  = (accountSize > 0) ? (totalRiskMoney / accountSize * 100.0) : 0;
   g_portfolioRisk.currentRewardMoney  = totalRewardMoney;
   g_portfolioRisk.currentRewardPercent= (accountSize > 0) ? (totalRewardMoney / accountSize * 100.0) : 0;
   g_portfolioRisk.currentLots         = totalLots;

   //--- Potential = current + calculated position
   double addRisk   = g_calcResult.riskMoneyResult;
   double addReward = g_calcResult.rewardMoneyResult;
   double addLots   = g_calcResult.lotSizeAdjusted;

   g_portfolioRisk.potentialRiskMoney    = totalRiskMoney + addRisk;
   g_portfolioRisk.potentialRiskPercent  = (accountSize > 0) ? (g_portfolioRisk.potentialRiskMoney / accountSize * 100.0) : 0;
   g_portfolioRisk.potentialRewardMoney  = totalRewardMoney + addReward;
   g_portfolioRisk.potentialRewardPercent= (accountSize > 0) ? (g_portfolioRisk.potentialRewardMoney / accountSize * 100.0) : 0;
   g_portfolioRisk.potentialLots         = totalLots + addLots;

   //--- R:R ratios
   g_portfolioRisk.currentRRRatio  = (totalRiskMoney > 0) ? (totalRewardMoney / totalRiskMoney) : 0;
   g_portfolioRisk.potentialRRRatio= (g_portfolioRisk.potentialRiskMoney > 0)
      ? (g_portfolioRisk.potentialRewardMoney / g_portfolioRisk.potentialRiskMoney) : 0;
}

#endif // CALCULATOR_MQH
