//+------------------------------------------------------------------+
//|                                                   ChartLines.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef CHART_LINES_MQH
#define CHART_LINES_MQH

#include "Defines.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"
#include "Calculator.mqh"

#define LINE_ENTRY       "LineEntry"
#define LINE_SL          "LineSL"
#define LINE_TP          "LineTP"
#define LINE_STOP_PRICE  "LineStopPrice"
#define LABEL_SL         "LblSL"
#define LABEL_TP         "LblTP"
#define LABEL_SL_INFO    "LblSLInfo"
#define LABEL_TP_INFO    "LblTPInfo"
#define LABEL_ENTRY_INFO "LblEntryInfo"

//+------------------------------------------------------------------+
void CreateOrUpdateHLine(string baseName, double price, color clr,
                         ENUM_LINE_STYLE style, int width, bool sel)
{
   string name = ObjName(baseName);
   if(ObjectFind(0, name) >= 0)
   {
      ObjectSetDouble(0, name, OBJPROP_PRICE, price);
      ObjectSetInteger(0, name, OBJPROP_COLOR, clr);
   }
   else
   {
      ObjectCreate(0, name, OBJ_HLINE, 0, 0, price);
      ObjectSetInteger(0, name, OBJPROP_COLOR, clr);
      ObjectSetInteger(0, name, OBJPROP_STYLE, style);
      ObjectSetInteger(0, name, OBJPROP_WIDTH, width);
      ObjectSetInteger(0, name, OBJPROP_SELECTABLE, sel);
      ObjectSetInteger(0, name, OBJPROP_SELECTED, sel && InpLinesSelected);
      ObjectSetInteger(0, name, OBJPROP_HIDDEN, false);
      ObjectSetInteger(0, name, OBJPROP_BACK, false);
   }
}

//+------------------------------------------------------------------+
void DeleteHLine(string baseName)
{ ObjectDelete(0, ObjName(baseName)); }

//+------------------------------------------------------------------+
void CreateOrUpdateLineLabel(string baseName, string text, double price, color clr)
{
   string name = ObjName(baseName);
   datetime t = TimeCurrent() + 5 * PeriodSeconds();
   if(ObjectFind(0, name) >= 0)
   {
      ObjectSetString(0, name, OBJPROP_TEXT, text);
      ObjectMove(0, name, 0, t, price);
      ObjectSetInteger(0, name, OBJPROP_COLOR, clr);
   }
   else
   {
      ObjectCreate(0, name, OBJ_TEXT, 0, t, price);
      ObjectSetString(0, name, OBJPROP_TEXT, text);
      ObjectSetInteger(0, name, OBJPROP_COLOR, clr);
      ObjectSetInteger(0, name, OBJPROP_FONTSIZE, InpLabelFontSize);
      ObjectSetString(0, name, OBJPROP_FONT, InpLabelFontFace);
      ObjectSetInteger(0, name, OBJPROP_ANCHOR, ANCHOR_LEFT_UPPER);
      ObjectSetInteger(0, name, OBJPROP_SELECTABLE, false);
      ObjectSetInteger(0, name, OBJPROP_HIDDEN, true);
      ObjectSetInteger(0, name, OBJPROP_BACK, InpDrawLabelsAsBackground);
   }
}

//+------------------------------------------------------------------+
void DeleteLineLabel(string baseName)
{ ObjectDelete(0, ObjName(baseName)); }

//+------------------------------------------------------------------+
void InitChartLines()
{
   double entry = GetEntryPrice();
   
   if(g_stopLossPrice == 0)
   {
      double dist = (InpDefaultSL > 0) ? InpDefaultSL : 500; // Default 500 points if not set
      double d = PointsToPrice((int)dist);
      g_stopLossPrice = (g_tradeDir == TRADE_DIR_LONG)
         ? NormalizePrice(entry - d) : NormalizePrice(entry + d);
   }
   
   if(g_takeProfitPrice == 0)
   {
      double dist = (InpDefaultTP > 0) ? InpDefaultTP : 1000; // Default 1000 points if not set
      double d = PointsToPrice((int)dist);
      g_takeProfitPrice = (g_tradeDir == TRADE_DIR_LONG)
         ? NormalizePrice(entry + d) : NormalizePrice(entry - d);
      
      if(g_tpCount > 0) g_tpLevels[0].price = g_takeProfitPrice;
   }
   
   if(g_linesVisible) DrawAllLines();
}

//+------------------------------------------------------------------+
void DrawAllLines()
{
   double entry = GetEntryPrice();
   if(!(InpHideEntryOnInstant && g_orderMode == ORDER_MODE_INSTANT))
      CreateOrUpdateHLine(LINE_ENTRY, entry, InpEntryLineColor,
         InpEntryLineStyle, InpEntryLineWidth, g_orderMode != ORDER_MODE_INSTANT);
   else DeleteHLine(LINE_ENTRY);

   if(g_stopLossPrice > 0)
      CreateOrUpdateHLine(LINE_SL, g_stopLossPrice, InpSLLineColor, InpSLLineStyle, InpSLLineWidth, true);
   if(g_takeProfitPrice > 0)
      CreateOrUpdateHLine(LINE_TP, g_takeProfitPrice, InpTPLineColor, InpTPLineStyle, InpTPLineWidth, true);

   if(g_orderMode == ORDER_MODE_STOP_LIMIT && g_stopPrice > 0)
      CreateOrUpdateHLine(LINE_STOP_PRICE, g_stopPrice, InpStopPriceLineColor, InpStopPriceStyle, InpStopPriceLineWidth, true);
   else DeleteHLine(LINE_STOP_PRICE);

   for(int i = 1; i < g_tpCount; i++)
   {
      string n = LINE_TP + IntegerToString(i + 1);
      if(g_tpLevels[i].price > 0)
         CreateOrUpdateHLine(n, g_tpLevels[i].price, InpTPLineColor, InpTPLineStyle, InpTPLineWidth, true);
      else DeleteHLine(n);
   }
   UpdateLineLabels();
   ChartRedraw(0);
}

//+------------------------------------------------------------------+
void UpdateLineLabels()
{
   if(!InpShowLineLabels && !InpShowSLInfoLabel && !InpShowTPInfoLabel && !InpShowEntryLabel) return;
   if(InpShowLineLabels && g_stopLossPrice > 0)
      CreateOrUpdateLineLabel(LABEL_SL, IntegerToString(CalcSLDistancePoints()) + " pts", g_stopLossPrice, InpSLLabelColor);
   else DeleteLineLabel(LABEL_SL);

   if(InpShowSLInfoLabel && g_stopLossPrice > 0)
      CreateOrUpdateLineLabel(LABEL_SL_INFO, FormatPercent(g_calcResult.riskPercentResult) + " | " + FormatMoneyShort(g_calcResult.riskMoneyResult), g_stopLossPrice, InpSLLabelColor);
   else DeleteLineLabel(LABEL_SL_INFO);

   if(InpShowLineLabels && g_takeProfitPrice > 0)
      CreateOrUpdateLineLabel(LABEL_TP, IntegerToString(CalcTPDistancePoints()) + " pts", g_takeProfitPrice, InpTPLabelColor);
   else DeleteLineLabel(LABEL_TP);

   if(InpShowTPInfoLabel && g_takeProfitPrice > 0)
      CreateOrUpdateLineLabel(LABEL_TP_INFO, FormatDouble(g_calcResult.rewardRiskResult, 2) + " R:R | " + FormatMoneyShort(g_calcResult.rewardMoneyResult), g_takeProfitPrice, InpTPLabelColor);
   else DeleteLineLabel(LABEL_TP_INFO);

   if(InpShowEntryLabel && GetEntryPrice() > 0)
      CreateOrUpdateLineLabel(LABEL_ENTRY_INFO, FormatDouble(g_calcResult.lotSizeAdjusted, 2) + " lots", GetEntryPrice(), InpEntryLabelColor);
   else DeleteLineLabel(LABEL_ENTRY_INFO);
}

//+------------------------------------------------------------------+
void RemoveAllLines()
{
   DeleteHLine(LINE_ENTRY); DeleteHLine(LINE_SL); DeleteHLine(LINE_TP); DeleteHLine(LINE_STOP_PRICE);
   DeleteLineLabel(LABEL_SL); DeleteLineLabel(LABEL_TP); DeleteLineLabel(LABEL_SL_INFO);
   DeleteLineLabel(LABEL_TP_INFO); DeleteLineLabel(LABEL_ENTRY_INFO);
   for(int i = 1; i < 20; i++) { DeleteHLine(LINE_TP + IntegerToString(i+1)); }
   ChartRedraw(0);
}

//+------------------------------------------------------------------+
void ToggleLinesVisibility()
{
   g_linesVisible = !g_linesVisible;
   g_panelState.linesVisible = g_linesVisible;
   if(g_linesVisible) DrawAllLines(); else RemoveAllLines();
}

//+------------------------------------------------------------------+
bool HandleLineDrag(string objectName)
{
   bool updated = false;
   if(objectName == ObjName(LINE_ENTRY) && g_orderMode != ORDER_MODE_INSTANT)
   {
      g_entryPrice = NormalizePrice(ObjectGetDouble(0, objectName, OBJPROP_PRICE));
      
      // Auto-switch direction if dragging Entry crosses a static SL
      if(!g_slInPoints && g_stopLossPrice > 0)
      {
         if(g_tradeDir == TRADE_DIR_LONG && g_stopLossPrice > g_entryPrice)
         {
            g_tradeDir = TRADE_DIR_SHORT;
            if(g_takeProfitPrice > 0) g_takeProfitPrice = NormalizePrice(g_entryPrice - MathAbs(g_takeProfitPrice - g_entryPrice));
         }
         else if(g_tradeDir == TRADE_DIR_SHORT && g_stopLossPrice < g_entryPrice)
         {
            g_tradeDir = TRADE_DIR_LONG;
            if(g_takeProfitPrice > 0) g_takeProfitPrice = NormalizePrice(g_entryPrice + MathAbs(g_takeProfitPrice - g_entryPrice));
         }
      }

      if(g_slInPoints && g_slPointsValue > 0)
      {
         double d = PointsToPrice(g_slPointsValue);
         g_stopLossPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(g_entryPrice - d) : NormalizePrice(g_entryPrice + d);
      }
      if(g_tpInPoints && g_tpPointsValue > 0)
      {
         double d = PointsToPrice(g_tpPointsValue);
         g_takeProfitPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(g_entryPrice + d) : NormalizePrice(g_entryPrice - d);
      }
      updated = true;
   }
   else if(objectName == ObjName(LINE_SL))
   {
      g_stopLossPrice = NormalizePrice(ObjectGetDouble(0, objectName, OBJPROP_PRICE));
      
      // Auto-switch direction based on dragged SL line
      double entry = GetEntryPrice();
      if(g_tradeDir == TRADE_DIR_LONG && g_stopLossPrice > entry)
      {
         g_tradeDir = TRADE_DIR_SHORT;
         if(g_takeProfitPrice > 0)
            g_takeProfitPrice = NormalizePrice(entry - MathAbs(g_takeProfitPrice - entry));
      }
      else if(g_tradeDir == TRADE_DIR_SHORT && g_stopLossPrice < entry)
      {
         g_tradeDir = TRADE_DIR_LONG;
         if(g_takeProfitPrice > 0)
            g_takeProfitPrice = NormalizePrice(entry + MathAbs(g_takeProfitPrice - entry));
      }
      
      if(g_slInPoints) g_slPointsValue = PriceToPoints(MathAbs(GetEntryPrice() - g_stopLossPrice));
      if(g_tpLockedOnSL && g_tpMultiplier > 0)
      {
         double entry = GetEntryPrice();
         double tpDist = MathAbs(entry - g_stopLossPrice) * g_tpMultiplier;
         g_takeProfitPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry + tpDist) : NormalizePrice(entry - tpDist);
      }
      updated = true;
   }
   else if(objectName == ObjName(LINE_TP))
   {
      g_takeProfitPrice = NormalizePrice(ObjectGetDouble(0, objectName, OBJPROP_PRICE));
      if(g_tpCount > 0) g_tpLevels[0].price = g_takeProfitPrice;
      if(g_tpInPoints) g_tpPointsValue = PriceToPoints(MathAbs(g_takeProfitPrice - GetEntryPrice()));
      updated = true;
   }
   else if(objectName == ObjName(LINE_STOP_PRICE))
   {
      g_stopPrice = NormalizePrice(ObjectGetDouble(0, objectName, OBJPROP_PRICE));
      updated = true;
   }
   else
   {
      for(int i = 1; i < g_tpCount; i++)
      {
         if(objectName == ObjName(LINE_TP + IntegerToString(i+1)))
         {
            g_tpLevels[i].price = NormalizePrice(ObjectGetDouble(0, objectName, OBJPROP_PRICE));
            updated = true; break;
         }
      }
   }
   return updated;
}

//+------------------------------------------------------------------+
void UpdateLinesOnTick()
{
   if(!g_linesVisible) return;
   if(g_orderMode == ORDER_MODE_INSTANT)
   {
      double entry = GetCurrentPrice();
      g_entryPrice = entry;
      if(g_slInPoints && g_slPointsValue > 0)
      {
         double d = PointsToPrice(g_slPointsValue);
         g_stopLossPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry - d) : NormalizePrice(entry + d);
      }
      if(g_tpInPoints && g_tpPointsValue > 0)
      {
         double d = PointsToPrice(g_tpPointsValue);
         g_takeProfitPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry + d) : NormalizePrice(entry - d);
      }
      DrawAllLines();
   }
   else { UpdateLineLabels(); ChartRedraw(0); }
}

#endif // CHART_LINES_MQH
