//+------------------------------------------------------------------+
//|                                                        Panel.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef PANEL_MQH
#define PANEL_MQH

#include "Defines.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"
#include "ChartLines.mqh"
#include "Settings.mqh"
#include "PanelMain.mqh"
#include "PanelRisk.mqh"
#include "PanelMargin.mqh"
#include "PanelSwaps.mqh"
#include "PanelTrading.mqh"

//--- Panel header/tab object names
#define PNL_BACKGROUND   "PnlBg"
#define PNL_HEADER       "PnlHeader"
#define PNL_TITLE        "PnlTitle"
#define PNL_DIR_ARROW    "PnlDirArrow"
#define PNL_BTN_MIN      "BtnMin"
#define PNL_BTN_CLOSE    "BtnClose"
#define PNL_TAB_MAIN     "TabMain"
#define PNL_TAB_RISK     "TabRisk"
#define PNL_TAB_MARGIN   "TabMargin"
#define PNL_TAB_SWAPS    "TabSwaps"
#define PNL_TAB_TRADING  "TabTrading"
#define PNL_TAB_ACCENT   "TabAccent"
#define PNL_MIN_BG       "MinBg"
#define PNL_MIN_TITLE    "MinTitle"
#define PNL_FOOTER_BG    "FooterBg"
#define PNL_FOOTER_TXT   "FooterTxt"
#define PNL_FOOTER_VER   "FooterVer"

//+------------------------------------------------------------------+
//| Get total panel height based on active tab                        |
//+------------------------------------------------------------------+
int GetPanelContentHeight()
{
   switch(g_activeTab)
   {
      case TAB_MAIN:    return g_mainTabHeight;
      case TAB_RISK:    return g_riskTabHeight;
      case TAB_MARGIN:  return g_marginTabHeight;
      case TAB_SWAPS:   return g_swapsTabHeight;
      case TAB_TRADING: return g_tradingTabHeight;
   }
   return g_mainTabHeight;
}

//+------------------------------------------------------------------+
//| Remove all panel tab content objects                               |
//+------------------------------------------------------------------+
void ClearTabContent()
{
   int total = ObjectsTotal(0, 0, -1);
   for(int i = total - 1; i >= 0; i--)
   {
      string name = ObjectName(0, i, 0, -1);
      if(StringFind(name, g_prefix) != 0) continue;
      // Keep panel frame objects
      if(StringFind(name, ObjName(PNL_BACKGROUND)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_HEADER)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TITLE)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_DIR_ARROW)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_BTN_MIN)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_BTN_CLOSE)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TAB_MAIN)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TAB_RISK)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TAB_MARGIN)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TAB_SWAPS)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TAB_TRADING)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_TAB_ACCENT)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_MIN_BG)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_MIN_TITLE)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_FOOTER_BG)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_FOOTER_TXT)) >= 0) continue;
      if(StringFind(name, ObjName(PNL_FOOTER_VER)) >= 0) continue;
      // Keep chart-anchored objects (lines and their labels)
      ENUM_OBJECT objType = (ENUM_OBJECT)ObjectGetInteger(0, name, OBJPROP_TYPE);
      if(objType == OBJ_HLINE || objType == OBJ_VLINE || objType == OBJ_TEXT) continue;
      ObjectDelete(0, name);
   }
}

//+------------------------------------------------------------------+
//| Create the full panel                                             |
//+------------------------------------------------------------------+
void CreatePanel()
{
   if(g_panelState.isMinimized)
   {
      CreateMinimizedPanel();
      return;
   }

   int px = g_panelState.posX;
   int py = g_panelState.posY;
   int tabW = (PANEL_WIDTH - 2) / 5;

   //--- Firstly run calculations so we know content height
   CalculatePositionSize();
   if(g_activeTab == TAB_RISK) CalculatePortfolioRisk();

   //--- Background (create first to stay at bottom)
   CreateRectLabel(0, ObjName(PNL_BACKGROUND), px, py, PANEL_WIDTH, 200,
      g_colors.clrBackground, g_colors.clrBorder, 1);

   //--- Header bar — premium deep look
   CreateRectLabel(0, ObjName(PNL_HEADER), px, py, PANEL_WIDTH, PANEL_HEADER_H,
      g_colors.clrHeaderBg, g_colors.clrHeaderBg, 0);

   //--- Direction indicator (subtle colored dot-arrow)
   string arrow = (g_tradeDir == TRADE_DIR_LONG) ? "▲" : "▼";
   color arrowClr = (g_tradeDir == TRADE_DIR_LONG) ? C'100,220,130' : C'255,100,110';
   CreateLabel(0, ObjName(PNL_DIR_ARROW), px + 8, py + 7, arrow, arrowClr, 11, PANEL_TITLE_FONT);

   //--- Title — company branding
   CreateLabel(0, ObjName(PNL_TITLE), px + 24, py + 8,
      "BrokersDB  Position Sizer", g_colors.clrHeaderText, 10, PANEL_TITLE_FONT);

   //--- Minimize button (sleek)
   CreateButton(0, ObjName(PNL_BTN_MIN), px + PANEL_WIDTH - 46, py + 3, 20, PANEL_HEADER_H - 6,
      "—", g_colors.clrHeaderText, g_colors.clrHeaderBg, 8);

   //--- Close button (sleek)
   CreateButton(0, ObjName(PNL_BTN_CLOSE), px + PANEL_WIDTH - 24, py + 3, 20, PANEL_HEADER_H - 6,
      "✕", g_colors.clrHeaderText, g_colors.clrHeaderBg, 8);

   //--- Tab background strip
   CreateRectLabel(0, ObjName("TabStrip"), px, py + PANEL_HEADER_H, PANEL_WIDTH, PANEL_TAB_H,
      g_colors.clrTabInactive, g_colors.clrBorder, 0);

   //--- Tab buttons
   int tabY = py + PANEL_HEADER_H;
   string tabNames[] = {"Main", "Risk", "Margin", "Swaps", "Trading"};
   string tabObjNames[] = {PNL_TAB_MAIN, PNL_TAB_RISK, PNL_TAB_MARGIN, PNL_TAB_SWAPS, PNL_TAB_TRADING};

   for(int i = 0; i < 5; i++)
   {
      int tw = (i < 4) ? tabW : PANEL_WIDTH - 4 * tabW;
      bool isActive = ((int)g_activeTab == i);
      color bg = isActive ? g_colors.clrTabActive : g_colors.clrTabInactive;
      color tc = isActive ? g_colors.clrEditText : g_colors.clrTabText;
      CreateButton(0, ObjName(tabObjNames[i]), px + i * tabW, tabY, tw, PANEL_TAB_H,
         tabNames[i], tc, bg, PANEL_FONT_SIZE);
   }

   //--- Active tab accent underline (2px colored bar)
   int accentX = px + (int)g_activeTab * tabW;
   int accentW = ((int)g_activeTab < 4) ? tabW : PANEL_WIDTH - 4 * tabW;
   CreateRectLabel(0, ObjName(PNL_TAB_ACCENT), accentX, tabY + PANEL_TAB_H - 2, accentW, 2,
      g_colors.clrTabAccent, g_colors.clrTabAccent, 0);

   //--- Create tab content
   ClearTabContent();
   switch(g_activeTab)
   {
      case TAB_MAIN:    CreateMainTab(); break;
      case TAB_RISK:    CreateRiskTab(); break;
      case TAB_MARGIN:  CreateMarginTab(); break;
      case TAB_SWAPS:   CreateSwapsTab(); break;
      case TAB_TRADING: CreateTradingTab(); break;
   }

   //--- Update background height to match content + footer
   int contentH = GetPanelContentHeight();
   int totalH = PANEL_HEADER_H + PANEL_TAB_H + contentH + PANEL_FOOTER_H + 2;
   ObjectSetInteger(0, ObjName(PNL_BACKGROUND), OBJPROP_YSIZE, totalH);

   //--- Footer bar — branding
   int footerY = py + PANEL_HEADER_H + PANEL_TAB_H + contentH;
   CreateRectLabel(0, ObjName(PNL_FOOTER_BG), px, footerY, PANEL_WIDTH, PANEL_FOOTER_H,
      g_colors.clrFooterBg, g_colors.clrBorder, 0);
   CreateLabel(0, ObjName(PNL_FOOTER_TXT), px + 10, footerY + 4,
      "brokersdb.com — Free Download", g_colors.clrFooterText, 7, PANEL_FONT_NAME);
   CreateLabel(0, ObjName(PNL_FOOTER_VER), px + PANEL_WIDTH - 35, footerY + 4,
      "v1.00", g_colors.clrFooterText, 7, PANEL_FONT_NAME);

   ChartRedraw(0);
}

//+------------------------------------------------------------------+
//| Moves the entire panel smoothly via XY distance deltas           |
//+------------------------------------------------------------------+
void MovePanelTo(int newX, int newY)
{
   if(newX == g_panelState.posX && newY == g_panelState.posY) return;

   int dx = newX - g_panelState.posX;
   int dy = newY - g_panelState.posY;
   
   g_panelState.posX = newX;
   g_panelState.posY = newY;
   
   int total = ObjectsTotal(0, 0, -1);
   for(int i = 0; i < total; i++)
   {
      string name = ObjectName(0, i, 0, -1);
      // Process only panel objects (prefix match)
      if(StringFind(name, g_prefix) != 0) continue;
      
      // Skip chart-anchored objects (OBJ_HLINE, OBJ_VLINE, OBJ_TEXT are bound to price/time)
      ENUM_OBJECT objType = (ENUM_OBJECT)ObjectGetInteger(0, name, OBJPROP_TYPE);
      if(objType == OBJ_HLINE || objType == OBJ_VLINE || objType == OBJ_TEXT) continue;
      
      long curX = ObjectGetInteger(0, name, OBJPROP_XDISTANCE);
      long curY = ObjectGetInteger(0, name, OBJPROP_YDISTANCE);
      
      ObjectSetInteger(0, name, OBJPROP_XDISTANCE, curX + dx);
      ObjectSetInteger(0, name, OBJPROP_YDISTANCE, curY + dy);
   }
   ChartRedraw(0);
}

//+------------------------------------------------------------------+
//| Create minimized panel — compact premium strip                    |
//+------------------------------------------------------------------+
void CreateMinimizedPanel()
{
   int px = g_panelState.posX;
   int py = g_panelState.posY;

   ClearTabContent();

   //--- Minimized background
   CreateRectLabel(0, ObjName(PNL_MIN_BG), px, py, 220, PANEL_HEADER_H,
      g_colors.clrHeaderBg, g_colors.clrBorder, 1);

   //--- Direction arrow
   string arrow = (g_tradeDir == TRADE_DIR_LONG) ? "▲" : "▼";
   color arrowClr = (g_tradeDir == TRADE_DIR_LONG) ? C'100,220,130' : C'255,100,110';
   CreateLabel(0, ObjName(PNL_DIR_ARROW), px + 8, py + 7, arrow, arrowClr, 10, PANEL_TITLE_FONT);

   //--- Title with lot size
   CreateLabel(0, ObjName(PNL_MIN_TITLE), px + 24, py + 8,
      "BDB PS: " + FormatDouble(g_calcResult.lotSizeAdjusted, 2) + " lots",
      g_colors.clrHeaderText, 9, PANEL_TITLE_FONT);

   //--- Maximize button
   CreateButton(0, ObjName(PNL_BTN_MIN), px + 192, py + 3, 22, PANEL_HEADER_H - 6,
      "□", g_colors.clrHeaderText, g_colors.clrHeaderBg, 9);

   ChartRedraw(0);
}

//+------------------------------------------------------------------+
//| Refresh panel (redraw current tab)                                |
//+------------------------------------------------------------------+
void RefreshPanel()
{
   CreatePanel();
}

//+------------------------------------------------------------------+
//| Destroy entire panel (preserves chart lines)                      |
//+------------------------------------------------------------------+
void DestroyPanel()
{
   //--- Delete only UI panel objects, preserve chart-anchored objects
   int total = ObjectsTotal(0, 0, -1);
   for(int i = total - 1; i >= 0; i--)
   {
      string name = ObjectName(0, i, 0, -1);
      if(StringFind(name, g_prefix) != 0) continue;
      //--- Keep chart-anchored objects (price lines & their text labels)
      ENUM_OBJECT objType = (ENUM_OBJECT)ObjectGetInteger(0, name, OBJPROP_TYPE);
      if(objType == OBJ_HLINE || objType == OBJ_VLINE || objType == OBJ_TEXT) continue;
      ObjectDelete(0, name);
   }
}

//+------------------------------------------------------------------+
//| Handle panel click events                                         |
//+------------------------------------------------------------------+
bool HandlePanelClick(string clickedObj)
{
   //--- Minimize/Maximize
   if(IsObjectClicked(clickedObj, PNL_BTN_MIN))
   {
      g_panelState.isMinimized = !g_panelState.isMinimized;
      DestroyPanel();
      CreatePanel();
      return true;
   }

   //--- Close
   if(IsObjectClicked(clickedObj, PNL_BTN_CLOSE))
   {
      if(InpAskBeforeClose)
      {
         int result = MessageBox("Remove BrokersDB Position Sizer from the chart?",
            "Confirm Close", MB_YESNO | MB_ICONQUESTION);
         if(result != IDYES) return true;
      }
      SaveSettings();
      ExpertRemove();
      return true;
   }

   //--- Tab switches
   if(IsObjectClicked(clickedObj, PNL_TAB_MAIN))    { SwitchTab(TAB_MAIN); return true; }
   if(IsObjectClicked(clickedObj, PNL_TAB_RISK))    { SwitchTab(TAB_RISK); return true; }
   if(IsObjectClicked(clickedObj, PNL_TAB_MARGIN))  { SwitchTab(TAB_MARGIN); return true; }
   if(IsObjectClicked(clickedObj, PNL_TAB_SWAPS))   { SwitchTab(TAB_SWAPS); return true; }
   if(IsObjectClicked(clickedObj, PNL_TAB_TRADING)) { SwitchTab(TAB_TRADING); return true; }

   //--- Delegate to active tab
   switch(g_activeTab)
   {
      case TAB_MAIN:    return HandleMainTabClick(clickedObj);
      case TAB_RISK:    return HandleRiskTabClick(clickedObj);
      case TAB_TRADING: return HandleTradingTabClick(clickedObj);
      default: break;
   }

   return false;
}

//+------------------------------------------------------------------+
//| Handle edit box change events                                     |
//+------------------------------------------------------------------+
bool HandlePanelEdit(string editObj)
{
   bool handled = false;
   switch(g_activeTab)
   {
      case TAB_MAIN:    handled = HandleMainTabEdit(editObj); break;
      case TAB_MARGIN:  handled = HandleMarginTabEdit(editObj); break;
      case TAB_TRADING: handled = HandleTradingTabEdit(editObj); break;
      default: break;
   }

   if(handled)
   {
      //--- Recalculate and refresh
      CalculatePositionSize();
      if(g_linesVisible) DrawAllLines();
      RefreshPanel();
   }
   return handled;
}

//+------------------------------------------------------------------+
//| Switch active tab                                                 |
//+------------------------------------------------------------------+
void SwitchTab(ENUM_PANEL_TAB newTab)
{
   if(newTab == g_activeTab) return;
   g_activeTab = newTab;
   g_panelState.activeTab = newTab;

   //--- Run calculations for the new tab
   if(newTab == TAB_RISK)    CalculatePortfolioRisk();
   if(newTab == TAB_MARGIN)  CalcMaxLotsByMargin();

   DestroyPanel();
   CreatePanel();
}

//+------------------------------------------------------------------+
//| Update only dynamic text of Panel objects, WITHOUT recreation     |
//+------------------------------------------------------------------+
void UpdatePanelData()
{
   if(g_panelState.isMinimized)
   {
      string title = "BDB PS: " + FormatDouble(g_calcResult.lotSizeAdjusted, 2) + " lots";
      ObjectSetString(0, ObjName(PNL_MIN_TITLE), OBJPROP_TEXT, title);
      return;
   }

   //--- Only update visible properties for the active tab to prevent flickering
   if(g_activeTab == TAB_MAIN)
   {
      double entryVal = GetEntryPrice();
      // If Instant mode, the entry price changes live
      if(g_orderMode == ORDER_MODE_INSTANT)
         ObjectSetString(0, ObjName(MT_EDT_ENTRY), OBJPROP_TEXT, FormatDouble(entryVal, GetSymbolDigits()));

      string slVal = g_slInPoints ? IntegerToString(g_slPointsValue) : FormatDouble(g_stopLossPrice, GetSymbolDigits());
      ObjectSetString(0, ObjName(MT_EDT_SL), OBJPROP_TEXT, slVal);

      string tpVal = g_tpInPoints ? IntegerToString(g_tpPointsValue) : FormatDouble(g_takeProfitPrice, GetSymbolDigits());
      ObjectSetString(0, ObjName(MT_EDT_TP), OBJPROP_TEXT, tpVal);

      // Live calculation results
      ObjectSetString(0, ObjName(MT_EDT_PS), OBJPROP_TEXT, FormatDouble(g_calcResult.lotSizeAdjusted, 2));
      ObjectSetString(0, ObjName(MT_LBL_RISK_R), OBJPROP_TEXT, FormatPercent(g_calcResult.riskPercentResult));
      ObjectSetString(0, ObjName(MT_LBL_RISKM_R), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.riskMoneyResult));
      
      if(g_takeProfitPrice > 0)
      {
         ObjectSetString(0, ObjName(MT_LBL_REWARD), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.rewardMoneyResult));
         ObjectSetString(0, ObjName(MT_LBL_REWARDR), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.rewardMoney));
         ObjectSetString(0, ObjName(MT_LBL_RR_IN), OBJPROP_TEXT, FormatDouble(g_calcResult.rewardRiskRatio, 2));
         ObjectSetString(0, ObjName(MT_LBL_RR_RES), OBJPROP_TEXT, "(adj: " + FormatDouble(g_calcResult.rewardRiskResult, 2) + ")");
      }
      if(InpShowPointValue)
      {
         ObjectSetString(0, ObjName(MT_LBL_PV), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.pointValue));
      }
   }
   else if(g_activeTab == TAB_RISK)
   {
      ObjectSetString(0, ObjName("ValCRiskM"), OBJPROP_TEXT, FormatMoneyShort(g_portfolioRisk.currentRiskMoney));
      ObjectSetString(0, ObjName("ValPRiskM"), OBJPROP_TEXT, FormatMoneyShort(g_portfolioRisk.potentialRiskMoney));
      ObjectSetString(0, ObjName("ValCRiskP"), OBJPROP_TEXT, FormatPercent(g_portfolioRisk.currentRiskPercent));
      ObjectSetString(0, ObjName("ValPRiskP"), OBJPROP_TEXT, FormatPercent(g_portfolioRisk.potentialRiskPercent));
      ObjectSetString(0, ObjName("ValCRewM"), OBJPROP_TEXT, FormatMoneyShort(g_portfolioRisk.currentRewardMoney));
      ObjectSetString(0, ObjName("ValPRewM"), OBJPROP_TEXT, FormatMoneyShort(g_portfolioRisk.potentialRewardMoney));
      ObjectSetString(0, ObjName("ValCRewP"), OBJPROP_TEXT, FormatPercent(g_portfolioRisk.currentRewardPercent));
      ObjectSetString(0, ObjName("ValPRewP"), OBJPROP_TEXT, FormatPercent(g_portfolioRisk.potentialRewardPercent));
      ObjectSetString(0, ObjName("ValCLots"), OBJPROP_TEXT, FormatDouble(g_portfolioRisk.currentLots, 2));
      ObjectSetString(0, ObjName("ValPLots"), OBJPROP_TEXT, FormatDouble(g_portfolioRisk.potentialLots, 2));
      ObjectSetString(0, ObjName("ValCRR"), OBJPROP_TEXT, FormatDouble(g_portfolioRisk.currentRRRatio, 2));
      ObjectSetString(0, ObjName("ValPRR"), OBJPROP_TEXT, FormatDouble(g_portfolioRisk.potentialRRRatio, 2));
   }
   else if(g_activeTab == TAB_MARGIN)
   {
      ObjectSetString(0, ObjName("ValPosMargin"), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.positionMargin));
      ObjectSetString(0, ObjName("ValFutUsed"), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.futureUsedMargin));
      ObjectSetString(0, ObjName("ValFutFree"), OBJPROP_TEXT, FormatMoneyShort(g_calcResult.futureFreeMargin));
   }
}

#endif // PANEL_MQH
