//+------------------------------------------------------------------+
//|                                                    PanelMain.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef PANEL_MAIN_MQH
#define PANEL_MAIN_MQH

#include "Defines.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"
#include "Calculator.mqh"
#include "ChartLines.mqh"
#include "TradeManager.mqh"

//--- Main tab object names
#define MT_BTN_DIR      "BtnDir"
#define MT_BTN_ORDTYPE  "BtnOrdType"
#define MT_BTN_LINES    "BtnLines"
#define MT_BTN_ACCT     "BtnAcct"
#define MT_BTN_QR1      "BtnQR1"
#define MT_BTN_QR2      "BtnQR2"
#define MT_BTN_MAXPS    "BtnMaxPS"
#define MT_BTN_TPMULT   "BtnTPMult"
#define MT_BTN_TPLOCK   "BtnTPLock"
#define MT_BTN_COMM_U   "BtnCommU"
#define MT_BTN_SL_SET   "BtnSLSet"
#define MT_BTN_ADDTP    "BtnAddTP"
#define MT_BTN_REMTP    "BtnRemTP"
#define MT_BTN_TRADE_M  "BtnTradeM"

#define MT_EDT_ENTRY    "EdtEntry"
#define MT_EDT_SL       "EdtSL"
#define MT_EDT_TP       "EdtTP"
#define MT_EDT_STOPPX   "EdtStopPx"
#define MT_EDT_COMM     "EdtComm"
#define MT_EDT_ACCT     "EdtAcct"
#define MT_EDT_RISK_P   "EdtRiskP"
#define MT_EDT_RISK_M   "EdtRiskM"
#define MT_EDT_PS       "EdtPS"
#define MT_EDT_ATRP     "EdtATRPer"
#define MT_EDT_ATRSL    "EdtATRSL"
#define MT_EDT_ATRTP    "EdtATRTP"

#define MT_LBL_SPREAD   "LblSpread"
#define MT_LBL_SL_D     "LblSLDist"
#define MT_LBL_TP_D     "LblTPDist"
#define MT_LBL_RISK_R   "LblRiskR"
#define MT_LBL_RISKM_R  "LblRiskMR"
#define MT_LBL_REWARD   "LblReward"
#define MT_LBL_REWARDR  "LblRewardR"
#define MT_LBL_RR_IN    "LblRRIn"
#define MT_LBL_RR_RES   "LblRRRes"
#define MT_LBL_PV       "LblPV"
#define MT_LBL_ATR_VAL  "LblATRVal"
#define MT_LBL_ATR_TF   "LblATRTF"

//+------------------------------------------------------------------+
//| Create Main tab controls                                          |
//+------------------------------------------------------------------+
void CreateMainTab()
{
   int x = g_panelState.posX + PANEL_MARGIN;
   int y = g_panelState.posY + PANEL_HEADER_H + PANEL_TAB_H + PANEL_SPACING + 2;
   int w = PANEL_WIDTH - 2 * PANEL_MARGIN;
   int halfW = (w - PANEL_SPACING) / 2;
   int thirdW = (w - 2 * PANEL_SPACING) / 3;
   int h = PANEL_ROW_H;
   int row = 0;
   int lblCol = 120;  // Label column width

   //--- Row 0: Direction button + Spread
   CreateButton(0, ObjName(MT_BTN_DIR), x, y + row * (h + PANEL_SPACING),
      72, h, (g_tradeDir == TRADE_DIR_LONG) ? "▲  Long" : "▼  Short",
      clrWhite, (g_tradeDir == TRADE_DIR_LONG) ? InpLongButtonColor : InpShortButtonColor);

   if(InpShowSpread != SPREAD_HIDE)
   {
      string spreadText = "Spread: " + IntegerToString(GetSpreadPoints()) + " pts";
      CreateLabel(0, ObjName(MT_LBL_SPREAD), x + 78, y + row * (h + PANEL_SPACING) + 4,
         spreadText, g_colors.clrTabText, PANEL_FONT_SIZE - 1);
   }
   row++;

   //--- Row 1: Order type + Lines toggle + Trade button
   CreateButton(0, ObjName(MT_BTN_ORDTYPE), x, y + row * (h + PANEL_SPACING),
      85, h, GetOrderModeText(), g_colors.clrButtonText, g_colors.clrButtonBg);

   CreateButton(0, ObjName(MT_BTN_LINES), x + 90, y + row * (h + PANEL_SPACING),
      65, h, g_linesVisible ? "☰ Lines" : "☰ Lines ✗", g_colors.clrButtonText, g_colors.clrButtonBg);

   if(InpTradeButtons == TRADE_BTN_MAIN || InpTradeButtons == TRADE_BTN_BOTH)
   {
      CreateButton(0, ObjName(MT_BTN_TRADE_M), x + w - 70, y + row * (h + PANEL_SPACING),
         70, h, "TRADE", clrWhite, InpTradeButtonColor);
   }
   row++;

   //--- Separator — Entry Section
   CreateRectLabel(0, ObjName("SepEntry"), x, y + row * (h + PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
   row++;

   //--- Row: Entry
   CreateLabel(0, ObjName("LblEntry"), x, y + row * (h + PANEL_SPACING) + 4,
      "Entry:", g_colors.clrLabelText);
   bool entryReadOnly = (g_orderMode == ORDER_MODE_INSTANT);
   double entryVal = GetEntryPrice();
   CreateEdit(0, ObjName(MT_EDT_ENTRY), x + lblCol, y + row * (h + PANEL_SPACING),
      w - lblCol, h, FormatDouble(entryVal, GetSymbolDigits()),
      entryReadOnly ? g_colors.clrTabText : g_colors.clrEditText,
      entryReadOnly ? g_colors.clrBackground : g_colors.clrEditBg,
      g_colors.clrEditBorder, 0, "", CORNER_LEFT_UPPER, ALIGN_RIGHT, entryReadOnly);
   row++;

   //--- Row: Stop-Loss
   string slLabel = g_slInPoints ? "SL (pts):" : "Stop Loss:";
   CreateLabel(0, ObjName("LblSLLabel"), x, y + row * (h + PANEL_SPACING) + 4,
      slLabel, g_colors.clrLabelText);
   if(InpDefaultSL > 0)
   {
      CreateButton(0, ObjName(MT_BTN_SL_SET), x + 72, y + row * (h + PANEL_SPACING),
         28, h, "⟳", g_colors.clrButtonText, g_colors.clrButtonBg);
   }
   string slVal = g_slInPoints ? IntegerToString(g_slPointsValue) : FormatDouble(g_stopLossPrice, GetSymbolDigits());
   CreateEdit(0, ObjName(MT_EDT_SL), x + lblCol, y + row * (h + PANEL_SPACING),
      w - lblCol, h, slVal, g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
   row++;

   //--- Row: Take-Profit
   string tpLabel = g_tpInPoints ? "TP (pts):" : "Take Profit:";
   CreateLabel(0, ObjName("LblTPLabel"), x, y + row * (h + PANEL_SPACING) + 4,
      tpLabel, g_colors.clrLabelText);

   CreateButton(0, ObjName(MT_BTN_TPMULT), x + 80, y + row * (h + PANEL_SPACING),
      22, h, FormatDouble(g_tpMultiplier, 1), g_colors.clrButtonText, g_colors.clrButtonBg, PANEL_FONT_SIZE - 1);

   CreateButton(0, ObjName(MT_BTN_TPLOCK), x + 104, y + row * (h + PANEL_SPACING),
      16, h, g_tpLockedOnSL ? "🔒" : "🔓", g_colors.clrButtonText, g_colors.clrButtonBg, 7);

   string tpVal = g_tpInPoints ? IntegerToString(g_tpPointsValue) : FormatDouble(g_takeProfitPrice, GetSymbolDigits());
   int tpEditX = x + lblCol;
   int tpEditW = w - lblCol - 28;
   CreateEdit(0, ObjName(MT_EDT_TP), tpEditX, y + row * (h + PANEL_SPACING),
      tpEditW, h, tpVal, g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);

   //--- Add/Remove TP buttons (compact)
   if(g_tpCount > 1)
      CreateButton(0, ObjName(MT_BTN_REMTP), x + w - 26, y + row * (h + PANEL_SPACING), 26, h, "−", g_colors.clrNegative, g_colors.clrButtonBg);
   CreateButton(0, ObjName(MT_BTN_ADDTP), x + w - (g_tpCount > 1 ? 52 : 26), y + row * (h + PANEL_SPACING), 26, h, "+", g_colors.clrPositive, g_colors.clrButtonBg);
   row++;

   //--- Additional TP levels
   for(int i = 1; i < g_tpCount; i++)
   {
      string tpN = "TP" + IntegerToString(i + 1) + ":";
      CreateLabel(0, ObjName("LblTP" + IntegerToString(i+1)), x, y + row * (h + PANEL_SPACING) + 4,
         tpN, g_colors.clrLabelText);
      CreateEdit(0, ObjName("EdtTP" + IntegerToString(i+1)), x + lblCol, y + row * (h + PANEL_SPACING),
         w - lblCol, h, FormatDouble(g_tpLevels[i].price, GetSymbolDigits()),
         g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
      row++;
   }

   //--- Stop Price (Stop Limit only)
   if(g_orderMode == ORDER_MODE_STOP_LIMIT)
   {
      CreateLabel(0, ObjName("LblStopPx"), x, y + row * (h + PANEL_SPACING) + 4,
         "Stop Price:", g_colors.clrLabelText);
      CreateEdit(0, ObjName(MT_EDT_STOPPX), x + lblCol, y + row * (h + PANEL_SPACING),
         w - lblCol, h, FormatDouble(g_stopPrice, GetSymbolDigits()),
         g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
      row++;
   }

   //--- ATR section
   if(InpShowATROptions)
   {
      CreateRectLabel(0, ObjName("SepATR"), x, y + row * (h + PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
      row++;

      CreateLabel(0, ObjName("LblATRPer"), x, y + row * (h + PANEL_SPACING) + 4, "ATR Period:", g_colors.clrLabelText);
      CreateEdit(0, ObjName(MT_EDT_ATRP), x + 80, y + row * (h + PANEL_SPACING), 45, h,
         IntegerToString(g_atrPeriod), g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
      CreateLabel(0, ObjName(MT_LBL_ATR_VAL), x + 130, y + row * (h + PANEL_SPACING) + 4,
         "ATR = " + FormatDouble(g_atrValue, GetSymbolDigits()), g_colors.clrResultText);
      row++;

      CreateLabel(0, ObjName("LblATRSLM"), x, y + row * (h + PANEL_SPACING) + 4, "SL ×:", g_colors.clrLabelText);
      CreateEdit(0, ObjName(MT_EDT_ATRSL), x + 40, y + row * (h + PANEL_SPACING), 55, h,
         FormatDouble(g_atrMultiplierSL, 2), g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
      CreateLabel(0, ObjName("LblATRTPM"), x + 105, y + row * (h + PANEL_SPACING) + 4, "TP ×:", g_colors.clrLabelText);
      CreateEdit(0, ObjName(MT_EDT_ATRTP), x + 145, y + row * (h + PANEL_SPACING), 55, h,
         FormatDouble(g_atrMultiplierTP, 2), g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
      row++;
   }

   //--- Separator — Risk Section
   CreateRectLabel(0, ObjName("SepMain1"), x, y + row * (h + PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
   row++;

   //--- Commission
   CreateLabel(0, ObjName("LblComm"), x, y + row * (h + PANEL_SPACING) + 4, "Commission:", g_colors.clrLabelText);
   CreateEdit(0, ObjName(MT_EDT_COMM), x + 95, y + row * (h + PANEL_SPACING), 65, h,
      FormatDouble(g_commission, 2), g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
   CreateButton(0, ObjName(MT_BTN_COMM_U), x + 165, y + row * (h + PANEL_SPACING), 32, h,
      (g_commType == COMMISSION_CURRENCY) ? "$" : "%", g_colors.clrButtonText, g_colors.clrButtonBg);
   row++;

   //--- Account size
   if(!InpHideAccountSize)
   {
      CreateButton(0, ObjName(MT_BTN_ACCT), x, y + row * (h + PANEL_SPACING), 80, h,
         GetAccountButtonText(), g_colors.clrButtonText, g_colors.clrButtonBg);

      string acctText = FormatMoneyShort(g_accountSize);
      if(InpAdditionalFunds != 0 || InpCustomBalance > 0) acctText += " *";
      CreateEdit(0, ObjName(MT_EDT_ACCT), x + 85, y + row * (h + PANEL_SPACING),
         w - 85, h, acctText, g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder, 0, "", CORNER_LEFT_UPPER, ALIGN_RIGHT, true);
      row++;
   }

   //--- Risk inputs row
   int riskRowY = y + row * (h + PANEL_SPACING);
   int riskX = x;

   if(InpQuickRisk1 > 0)
   {
      CreateButton(0, ObjName(MT_BTN_QR1), riskX, riskRowY, 38, h,
         FormatDouble(InpQuickRisk1, 1) + "%", g_colors.clrButtonText, g_colors.clrButtonBg);
      riskX += 40;
   }
   if(InpQuickRisk2 > 0)
   {
      CreateButton(0, ObjName(MT_BTN_QR2), riskX, riskRowY, 38, h,
         FormatDouble(InpQuickRisk2, 1) + "%", g_colors.clrButtonText, g_colors.clrButtonBg);
      riskX += 40;
   }

   CreateLabel(0, ObjName("LblRisk"), riskX, riskRowY + 4, "Risk %:", g_colors.clrLabelText);
   int riskEditW = 60;
   CreateEdit(0, ObjName(MT_EDT_RISK_P), riskX + 50, riskRowY, riskEditW, h,
      FormatDouble(g_riskPercent, 2), g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);

   CreateLabel(0, ObjName("LblRiskM"), riskX + 50 + riskEditW + 5, riskRowY + 4, "$:", g_colors.clrLabelText);
   double moneyRiskDisplay = (g_riskMoney > 0) ? g_riskMoney : (g_accountSize * g_riskPercent / 100.0);
   CreateEdit(0, ObjName(MT_EDT_RISK_M), x + w - 75, riskRowY, 75, h,
      FormatMoneyShort(moneyRiskDisplay), g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
   row++;

   //--- Separator — Results Section  
   CreateRectLabel(0, ObjName("SepMain2"), x, y + row * (h + PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
   row++;

   //--- Risk result
   CreateLabel(0, ObjName("LblRiskRes"), x, y + row * (h + PANEL_SPACING) + 4, "Risk:", g_colors.clrLabelText);
   CreateLabel(0, ObjName(MT_LBL_RISK_R), x + 55, y + row * (h + PANEL_SPACING) + 4,
      FormatPercent(g_calcResult.riskPercentResult), g_colors.clrNegative);
   CreateLabel(0, ObjName(MT_LBL_RISKM_R), x + 130, y + row * (h + PANEL_SPACING) + 4,
      FormatMoneyShort(g_calcResult.riskMoneyResult), g_colors.clrNegative);
   row++;

   //--- Reward result  
   if(g_takeProfitPrice > 0)
   {
      CreateLabel(0, ObjName("LblRewRes"), x, y + row * (h + PANEL_SPACING) + 4, "Reward:", g_colors.clrLabelText);
      CreateLabel(0, ObjName(MT_LBL_REWARD), x + 55, y + row * (h + PANEL_SPACING) + 4,
         FormatMoneyShort(g_calcResult.rewardMoneyResult), g_colors.clrPositive);
      CreateLabel(0, ObjName(MT_LBL_REWARDR), x + 150, y + row * (h + PANEL_SPACING) + 4,
         FormatMoneyShort(g_calcResult.rewardMoney), g_colors.clrPositive);
      row++;

      //--- R:R ratio
      CreateLabel(0, ObjName("LblRRLabel"), x, y + row * (h + PANEL_SPACING) + 4, "R:R Ratio:", g_colors.clrLabelText);
      CreateLabel(0, ObjName(MT_LBL_RR_IN), x + 75, y + row * (h + PANEL_SPACING) + 4,
         FormatDouble(g_calcResult.rewardRiskRatio, 2), g_colors.clrResultText);
      CreateLabel(0, ObjName(MT_LBL_RR_RES), x + 130, y + row * (h + PANEL_SPACING) + 4,
         "(adj: " + FormatDouble(g_calcResult.rewardRiskResult, 2) + ")", g_colors.clrTabText);
      row++;
   }

   //--- Separator — Position
   CreateRectLabel(0, ObjName("SepMain3"), x, y + row * (h + PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
   row++;

   //--- Position Size (editable)
   CreateLabel(0, ObjName("LblPSLabel"), x, y + row * (h + PANEL_SPACING) + 4, "Position Size:", g_colors.clrLabelText, PANEL_FONT_SIZE, PANEL_TITLE_FONT);
   int psEditW = w - lblCol;
   if(InpShowMaxPSButton) psEditW -= 38;
   CreateEdit(0, ObjName(MT_EDT_PS), x + lblCol, y + row * (h + PANEL_SPACING),
      psEditW, h, FormatDouble(g_calcResult.lotSizeAdjusted, 2),
      g_colors.clrEditText, g_colors.clrEditBg, g_colors.clrEditBorder);
   if(InpShowMaxPSButton)
   {
      CreateButton(0, ObjName(MT_BTN_MAXPS), x + w - 35, y + row * (h + PANEL_SPACING),
         35, h, "Max", clrWhite, g_colors.clrBtnTrade);
   }
   row++;

   //--- Point Value
   if(InpShowPointValue)
   {
      CreateLabel(0, ObjName("LblPVLabel"), x, y + row * (h + PANEL_SPACING) + 4, "Point Value:", g_colors.clrLabelText);
      CreateLabel(0, ObjName(MT_LBL_PV), x + lblCol, y + row * (h + PANEL_SPACING) + 4,
         FormatMoneyShort(g_calcResult.pointValue), g_colors.clrResultText);
      row++;
   }

   //--- Store calculated height
   g_mainTabHeight = row * (h + PANEL_SPACING) + PANEL_SPACING + 2;
}

//+------------------------------------------------------------------+
//| Get order mode text string                                        |
//+------------------------------------------------------------------+
string GetOrderModeText()
{
   switch(g_orderMode)
   {
      case ORDER_MODE_INSTANT:    return "Instant";
      case ORDER_MODE_PENDING:    return "Pending";
      case ORDER_MODE_STOP_LIMIT: return "Stop Limit";
   }
   return "Instant";
}

//+------------------------------------------------------------------+
//| Get account button text string                                    |
//+------------------------------------------------------------------+
string GetAccountButtonText()
{
   switch(g_accountBase)
   {
      case ACCT_BALANCE:           return "Balance";
      case ACCT_EQUITY:            return "Equity";
      case ACCT_BALANCE_MINUS_CPR: return "Bal - CPR";
   }
   return "Balance";
}

//+------------------------------------------------------------------+
//| Handle Main tab click events                                      |
//+------------------------------------------------------------------+
bool HandleMainTabClick(string clickedObj)
{
   //--- Direction toggle
   if(IsObjectClicked(clickedObj, MT_BTN_DIR))
   {
      g_tradeDir = (g_tradeDir == TRADE_DIR_LONG) ? TRADE_DIR_SHORT : TRADE_DIR_LONG;
      //--- Swap SL/TP sides
      if(g_stopLossPrice > 0 && g_takeProfitPrice > 0)
      {
         double entry = GetEntryPrice();
         double slDist = MathAbs(entry - g_stopLossPrice);
         double tpDist = MathAbs(g_takeProfitPrice - entry);
         if(g_tradeDir == TRADE_DIR_LONG)
         {
            g_stopLossPrice = NormalizePrice(entry - slDist);
            g_takeProfitPrice = NormalizePrice(entry + tpDist);
         }
         else
         {
            g_stopLossPrice = NormalizePrice(entry + slDist);
            g_takeProfitPrice = NormalizePrice(entry - tpDist);
         }
      }
      return true;
   }

   //--- Order type toggle
   if(IsObjectClicked(clickedObj, MT_BTN_ORDTYPE))
   {
      SwitchOrderMode();
      return true;
   }

   //--- Lines toggle
   if(IsObjectClicked(clickedObj, MT_BTN_LINES))
   {
      ToggleLinesVisibility();
      return true;
   }

   //--- Account button
   if(IsObjectClicked(clickedObj, MT_BTN_ACCT))
   {
      g_accountBase = (ENUM_ACCOUNT_BUTTON)(((int)g_accountBase + 1) % 3);
      g_accountSize = GetAccountSize();
      return true;
   }

   //--- Quick risk buttons
   if(IsObjectClicked(clickedObj, MT_BTN_QR1) && InpQuickRisk1 > 0)
   {
      g_riskPercent = InpQuickRisk1;
      g_riskMoney = 0;
      return true;
   }
   if(IsObjectClicked(clickedObj, MT_BTN_QR2) && InpQuickRisk2 > 0)
   {
      g_riskPercent = InpQuickRisk2;
      g_riskMoney = 0;
      return true;
   }

   //--- Max PS button
   if(IsObjectClicked(clickedObj, MT_BTN_MAXPS))
   {
      double maxLots = CalcMaxLotsByMargin();
      if(maxLots > 0)
      {
         CalcRiskFromPositionSize(maxLots);
      }
      return true;
   }

   //--- Commission units toggle
   if(IsObjectClicked(clickedObj, MT_BTN_COMM_U))
   {
      g_commType = (g_commType == COMMISSION_CURRENCY) ? COMMISSION_PERCENT : COMMISSION_CURRENCY;
      return true;
   }

   //--- TP multiplier button - apply TP = SL * multiplier
   if(IsObjectClicked(clickedObj, MT_BTN_TPMULT))
   {
      double entry = GetEntryPrice();
      double slDist = MathAbs(entry - g_stopLossPrice);
      double tpDist = slDist * g_tpMultiplier;
      if(g_tradeDir == TRADE_DIR_LONG)
         g_takeProfitPrice = NormalizePrice(entry + tpDist);
      else
         g_takeProfitPrice = NormalizePrice(entry - tpDist);
      if(g_tpCount > 0) g_tpLevels[0].price = g_takeProfitPrice;
      return true;
   }

   //--- TP lock toggle
   if(IsObjectClicked(clickedObj, MT_BTN_TPLOCK))
   {
      g_tpLockedOnSL = !g_tpLockedOnSL;
      return true;
   }

   //--- SL set button
   if(IsObjectClicked(clickedObj, MT_BTN_SL_SET) && InpDefaultSL > 0)
   {
      double entry = GetEntryPrice();
      double d = PointsToPrice(InpDefaultSL);
      if(g_tradeDir == TRADE_DIR_LONG) g_stopLossPrice = NormalizePrice(entry - d);
      else g_stopLossPrice = NormalizePrice(entry + d);
      return true;
   }

   //--- Add TP
   if(IsObjectClicked(clickedObj, MT_BTN_ADDTP))
   {
      g_tpCount++;
      ArrayResize(g_tpLevels, g_tpCount);
      int idx = g_tpCount - 1;
      if(InpPrefillTPs && g_takeProfitPrice > 0)
      {
         double entry = GetEntryPrice();
         double tpDist = MathAbs(g_takeProfitPrice - entry);
         double newDist = tpDist * (1.0 + 0.5 * idx);
         g_tpLevels[idx].price = (g_tradeDir == TRADE_DIR_LONG)
            ? NormalizePrice(entry + newDist) : NormalizePrice(entry - newDist);
      }
      else g_tpLevels[idx].price = 0;

      g_tpLevels[idx].sharePct = 100.0 / g_tpCount;
      g_tpLevels[idx].lots = 0;
      //--- Redistribute shares equally
      for(int i = 0; i < g_tpCount; i++)
         g_tpLevels[i].sharePct = 100.0 / g_tpCount;
      return true;
   }

   //--- Remove TP
   if(IsObjectClicked(clickedObj, MT_BTN_REMTP) && g_tpCount > 1)
   {
      g_tpCount--;
      ArrayResize(g_tpLevels, g_tpCount);
      for(int i = 0; i < g_tpCount; i++)
         g_tpLevels[i].sharePct = 100.0 / g_tpCount;
      return true;
   }

   //--- Trade button on main tab
   if(IsObjectClicked(clickedObj, MT_BTN_TRADE_M))
   {
      ExecuteTrade();
      return true;
   }

   return false;
}

//+------------------------------------------------------------------+
//| Handle Main tab edit box value changes                            |
//+------------------------------------------------------------------+
bool HandleMainTabEdit(string editObj)
{
   int digits = GetSymbolDigits();

   if(IsObjectClicked(editObj, MT_EDT_ENTRY))
   {
      string text = ObjectGetString(0, ObjName(MT_EDT_ENTRY), OBJPROP_TEXT);
      g_entryPrice = NormalizePrice(StringToDouble(text));
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_SL))
   {
      string text = ObjectGetString(0, ObjName(MT_EDT_SL), OBJPROP_TEXT);
      if(g_slInPoints)
      {
         g_slPointsValue = (int)StringToInteger(text);
         double entry = GetEntryPrice();
         double d = PointsToPrice(g_slPointsValue);
         g_stopLossPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry - d) : NormalizePrice(entry + d);
      }
      else
         g_stopLossPrice = NormalizePrice(StringToDouble(text));
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_TP))
   {
      string text = ObjectGetString(0, ObjName(MT_EDT_TP), OBJPROP_TEXT);
      if(g_tpInPoints)
      {
         g_tpPointsValue = (int)StringToInteger(text);
         double entry = GetEntryPrice();
         double d = PointsToPrice(g_tpPointsValue);
         g_takeProfitPrice = (g_tradeDir == TRADE_DIR_LONG) ? NormalizePrice(entry + d) : NormalizePrice(entry - d);
      }
      else
         g_takeProfitPrice = NormalizePrice(StringToDouble(text));
      if(g_tpCount > 0) g_tpLevels[0].price = g_takeProfitPrice;
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_STOPPX))
   {
      g_stopPrice = NormalizePrice(StringToDouble(ObjectGetString(0, ObjName(MT_EDT_STOPPX), OBJPROP_TEXT)));
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_COMM))
   {
      g_commission = StringToDouble(ObjectGetString(0, ObjName(MT_EDT_COMM), OBJPROP_TEXT));
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_RISK_P))
   {
      g_riskPercent = StringToDouble(ObjectGetString(0, ObjName(MT_EDT_RISK_P), OBJPROP_TEXT));
      g_riskMoney = 0;
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_RISK_M))
   {
      g_riskMoney = StringToDouble(ObjectGetString(0, ObjName(MT_EDT_RISK_M), OBJPROP_TEXT));
      return true;
   }
   if(IsObjectClicked(editObj, MT_EDT_PS))
   {
      double ps = StringToDouble(ObjectGetString(0, ObjName(MT_EDT_PS), OBJPROP_TEXT));
      CalcRiskFromPositionSize(ps);
      return true;
   }
   if(InpShowATROptions)
   {
      if(IsObjectClicked(editObj, MT_EDT_ATRP))
      {
         g_atrPeriod = (int)StringToInteger(ObjectGetString(0, ObjName(MT_EDT_ATRP), OBJPROP_TEXT));
         ReleaseATR(); InitATR(); UpdateATRValue();
         return true;
      }
      if(IsObjectClicked(editObj, MT_EDT_ATRSL))
      {
         g_atrMultiplierSL = StringToDouble(ObjectGetString(0, ObjName(MT_EDT_ATRSL), OBJPROP_TEXT));
         return true;
      }
      if(IsObjectClicked(editObj, MT_EDT_ATRTP))
      {
         g_atrMultiplierTP = StringToDouble(ObjectGetString(0, ObjName(MT_EDT_ATRTP), OBJPROP_TEXT));
         return true;
      }
   }

   //--- Multi-TP edits
   for(int i = 1; i < g_tpCount; i++)
   {
      string tpEdit = "EdtTP" + IntegerToString(i+1);
      if(IsObjectClicked(editObj, tpEdit))
      {
         g_tpLevels[i].price = NormalizePrice(StringToDouble(ObjectGetString(0, ObjName(tpEdit), OBJPROP_TEXT)));
         return true;
      }
   }

   return false;
}

//+------------------------------------------------------------------+
//| Switch order mode cycling                                         |
//+------------------------------------------------------------------+
void SwitchOrderMode()
{
   if(g_orderMode == ORDER_MODE_INSTANT)
   {
      g_orderMode = ORDER_MODE_PENDING;
      g_entryPrice = GetCurrentPrice();
   }
   else if(g_orderMode == ORDER_MODE_PENDING)
   {
      if(!InpDisableStopLimit)
      {
         g_orderMode = ORDER_MODE_STOP_LIMIT;
         g_stopPrice = GetCurrentPrice();
      }
      else
         g_orderMode = ORDER_MODE_INSTANT;
   }
   else
   {
      g_orderMode = ORDER_MODE_INSTANT;
   }
}

//--- Global variable for main tab height
int g_mainTabHeight = 0;

#endif // PANEL_MAIN_MQH
