//+------------------------------------------------------------------+
//|                                                   PanelSwaps.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef PANEL_SWAPS_MQH
#define PANEL_SWAPS_MQH

#include "Defines.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"

int g_swapsTabHeight = 0;

//+------------------------------------------------------------------+
string GetSwapTypeText(ENUM_SYMBOL_SWAP_MODE mode)
{
   switch(mode)
   {
      case SYMBOL_SWAP_MODE_POINTS:           return "Points";
      case SYMBOL_SWAP_MODE_CURRENCY_SYMBOL:  return "Base Currency";
      case SYMBOL_SWAP_MODE_INTEREST_CURRENT: return "Interest";
      case SYMBOL_SWAP_MODE_CURRENCY_DEPOSIT: return "Account Currency";
      case SYMBOL_SWAP_MODE_CURRENCY_MARGIN:  return "Margin Currency";
      case SYMBOL_SWAP_MODE_REOPEN_CURRENT:   return "Reopen (Current)";
      case SYMBOL_SWAP_MODE_REOPEN_BID:       return "Reopen (Bid)";
      default: return "Disabled";
   }
}

//+------------------------------------------------------------------+
string GetTripleSwapDay(int day)
{
   switch(day)
   {
      case 0: return "Monday";
      case 1: return "Tuesday";
      case 2: return "Wednesday";
      case 3: return "Thursday";
      case 4: return "Friday";
      default: return "None";
   }
}

//+------------------------------------------------------------------+
double CalcDailySwapPerLot(double nominalSwap, string symbol)
{
   ENUM_SYMBOL_SWAP_MODE mode = (ENUM_SYMBOL_SWAP_MODE)SymbolInfoInteger(symbol, SYMBOL_SWAP_MODE);
   double tickValue = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
   double tickSize  = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE);
   double point     = SymbolInfoDouble(symbol, SYMBOL_POINT);

   switch(mode)
   {
      case SYMBOL_SWAP_MODE_POINTS:
         if(tickSize > 0) return nominalSwap * point * tickValue / tickSize;
         break;
      case SYMBOL_SWAP_MODE_CURRENCY_SYMBOL:
      case SYMBOL_SWAP_MODE_CURRENCY_DEPOSIT:
      case SYMBOL_SWAP_MODE_CURRENCY_MARGIN:
         return nominalSwap;
      case SYMBOL_SWAP_MODE_INTEREST_CURRENT:
      {
         double price = (GetBid(symbol) + GetAsk(symbol)) / 2.0;
         double contractSize = SymbolInfoDouble(symbol, SYMBOL_TRADE_CONTRACT_SIZE);
         return nominalSwap / 100.0 / 360.0 * price * contractSize * tickValue / tickSize;
      }
      default: break;
   }
   return 0;
}

//+------------------------------------------------------------------+
void CreateSwapsTab()
{
   int x = g_panelState.posX + PANEL_MARGIN;
   int y = g_panelState.posY + PANEL_HEADER_H + PANEL_TAB_H + PANEL_SPACING + 2;
   int w = PANEL_WIDTH - 2 * PANEL_MARGIN;
   int h = PANEL_ROW_H;
   int row = 0;
   string sym = GetTradeSymbol();
   int valX = x + 130;

   //--- Type
   ENUM_SYMBOL_SWAP_MODE swMode = (ENUM_SYMBOL_SWAP_MODE)SymbolInfoInteger(sym, SYMBOL_SWAP_MODE);
   CreateLabel(0, ObjName("LblSwType"), x, y+row*(h+PANEL_SPACING)+3, "Swap Type:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwType"), valX, y+row*(h+PANEL_SPACING)+3, GetSwapTypeText(swMode), g_colors.clrResultText);
   row++;

   //--- Triple Swap Day
   int tripleDay = (int)SymbolInfoInteger(sym, SYMBOL_SWAP_ROLLOVER3DAYS);
   CreateLabel(0, ObjName("LblSwTriple"), x, y+row*(h+PANEL_SPACING)+3, "Triple Swap:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwTriple"), valX, y+row*(h+PANEL_SPACING)+3, GetTripleSwapDay(tripleDay), g_colors.clrResultText);
   row++;

   //--- Separator
   CreateRectLabel(0, ObjName("SepSw1"), x, y+row*(h+PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
   row++;

   double swapLong  = SymbolInfoDouble(sym, SYMBOL_SWAP_LONG);
   double swapShort = SymbolInfoDouble(sym, SYMBOL_SWAP_SHORT);
   double lots = g_calcResult.lotSizeAdjusted;

   //--- Header
   CreateLabel(0, ObjName("LblSwHdL"), valX, y+row*(h+PANEL_SPACING)+3, "Long", g_colors.clrLabelText);
   CreateLabel(0, ObjName("LblSwHdS"), valX+70, y+row*(h+PANEL_SPACING)+3, "Short", g_colors.clrLabelText);
   row++;

   //--- Nominal
   CreateLabel(0, ObjName("LblSwNom"), x, y+row*(h+PANEL_SPACING)+3, "Nominal Swap:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwNomL"), valX, y+row*(h+PANEL_SPACING)+3, FormatDouble(swapLong, 2), swapLong>=0?g_colors.clrPositive:g_colors.clrNegative);
   CreateLabel(0, ObjName("ValSwNomS"), valX+70, y+row*(h+PANEL_SPACING)+3, FormatDouble(swapShort, 2), swapShort>=0?g_colors.clrPositive:g_colors.clrNegative);
   row++;

   double dailyL = CalcDailySwapPerLot(swapLong, sym);
   double dailyS = CalcDailySwapPerLot(swapShort, sym);

   //--- Daily per lot
   CreateLabel(0, ObjName("LblSwDLot"), x, y+row*(h+PANEL_SPACING)+3, "Daily/lot:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwDLotL"), valX, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyL), dailyL>=0?g_colors.clrPositive:g_colors.clrNegative);
   CreateLabel(0, ObjName("ValSwDLotS"), valX+70, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyS), dailyS>=0?g_colors.clrPositive:g_colors.clrNegative);
   row++;

   //--- Daily per PS
   CreateLabel(0, ObjName("LblSwDPS"), x, y+row*(h+PANEL_SPACING)+3, "Daily/PS:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwDPSL"), valX, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyL*lots), dailyL>=0?g_colors.clrPositive:g_colors.clrNegative);
   CreateLabel(0, ObjName("ValSwDPSS"), valX+70, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyS*lots), dailyS>=0?g_colors.clrPositive:g_colors.clrNegative);
   row++;

   //--- Yearly per lot
   CreateLabel(0, ObjName("LblSwYLot"), x, y+row*(h+PANEL_SPACING)+3, "Yearly/lot:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwYLotL"), valX, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyL*360), dailyL>=0?g_colors.clrPositive:g_colors.clrNegative);
   CreateLabel(0, ObjName("ValSwYLotS"), valX+70, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyS*360), dailyS>=0?g_colors.clrPositive:g_colors.clrNegative);
   row++;

   //--- Yearly per PS
   CreateLabel(0, ObjName("LblSwYPS"), x, y+row*(h+PANEL_SPACING)+3, "Yearly/PS:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwYPSL"), valX, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyL*lots*360), dailyL>=0?g_colors.clrPositive:g_colors.clrNegative);
   CreateLabel(0, ObjName("ValSwYPSS"), valX+70, y+row*(h+PANEL_SPACING)+3, FormatMoneyShort(dailyS*lots*360), dailyS>=0?g_colors.clrPositive:g_colors.clrNegative);
   row++;

   //--- Separator
   CreateRectLabel(0, ObjName("SepSw2"), x, y+row*(h+PANEL_SPACING), w, 1, g_colors.clrSeparator, g_colors.clrSeparator);
   row++;

   //--- Pos Size
   CreateLabel(0, ObjName("LblSwPS"), x, y+row*(h+PANEL_SPACING)+3, "Position Size:", g_colors.clrLabelText);
   CreateLabel(0, ObjName("ValSwPS"), valX, y+row*(h+PANEL_SPACING)+3, FormatDouble(lots, 2) + " lots", g_colors.clrResultText);
   row++;

   g_swapsTabHeight = row * (h + PANEL_SPACING) + PANEL_SPACING;
}

#endif // PANEL_SWAPS_MQH
