//+------------------------------------------------------------------+
//|                                                     Settings.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef SETTINGS_MQH
#define SETTINGS_MQH

#include "Defines.mqh"
#include "Globals.mqh"

//+------------------------------------------------------------------+
//| Get settings filename for current symbol                          |
//+------------------------------------------------------------------+
string GetSettingsFilePath(string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   return BDB_SETTINGS_DIR + "\\" + BDB_SETTINGS_FILE + "_" + symbol + ".txt";
}

//+------------------------------------------------------------------+
//| Get general settings filename (non-symbol-specific)               |
//+------------------------------------------------------------------+
string GetGeneralSettingsPath()
{
   return BDB_SETTINGS_DIR + "\\" + BDB_SETTINGS_FILE + "_general.txt";
}

//+------------------------------------------------------------------+
//| Save current panel settings to file                               |
//+------------------------------------------------------------------+
bool SaveSettings(string symbol = "")
{
   string filePath = GetSettingsFilePath(symbol);
   int handle = FileOpen(filePath, FILE_WRITE | FILE_TXT);
   if(handle == INVALID_HANDLE)
   {
      Print("[BrokersDB PS] Cannot save settings: ", filePath);
      return false;
   }

   FileWriteString(handle, "PanelX=" + IntegerToString(g_panelState.posX) + "\n");
   FileWriteString(handle, "PanelY=" + IntegerToString(g_panelState.posY) + "\n");
   FileWriteString(handle, "Minimized=" + IntegerToString(g_panelState.isMinimized) + "\n");
   FileWriteString(handle, "ActiveTab=" + IntegerToString((int)g_activeTab) + "\n");
   FileWriteString(handle, "LinesVisible=" + IntegerToString(g_linesVisible) + "\n");
   FileWriteString(handle, "TradeDir=" + IntegerToString((int)g_tradeDir) + "\n");
   FileWriteString(handle, "OrderMode=" + IntegerToString((int)g_orderMode) + "\n");
   FileWriteString(handle, "EntryPrice=" + DoubleToString(g_entryPrice, 8) + "\n");
   FileWriteString(handle, "SLPrice=" + DoubleToString(g_stopLossPrice, 8) + "\n");
   FileWriteString(handle, "TPPrice=" + DoubleToString(g_takeProfitPrice, 8) + "\n");
   FileWriteString(handle, "StopPrice=" + DoubleToString(g_stopPrice, 8) + "\n");
   FileWriteString(handle, "SLInPoints=" + IntegerToString(g_slInPoints) + "\n");
   FileWriteString(handle, "TPInPoints=" + IntegerToString(g_tpInPoints) + "\n");
   FileWriteString(handle, "SLPointsVal=" + IntegerToString(g_slPointsValue) + "\n");
   FileWriteString(handle, "TPPointsVal=" + IntegerToString(g_tpPointsValue) + "\n");
   FileWriteString(handle, "RiskPercent=" + DoubleToString(g_riskPercent, 4) + "\n");
   FileWriteString(handle, "RiskMoney=" + DoubleToString(g_riskMoney, 2) + "\n");
   FileWriteString(handle, "AccountBase=" + IntegerToString((int)g_accountBase) + "\n");
   FileWriteString(handle, "Commission=" + DoubleToString(g_commission, 4) + "\n");
   FileWriteString(handle, "CommType=" + IntegerToString((int)g_commType) + "\n");
   FileWriteString(handle, "ATRPeriod=" + IntegerToString(g_atrPeriod) + "\n");
   FileWriteString(handle, "ATRMultSL=" + DoubleToString(g_atrMultiplierSL, 2) + "\n");
   FileWriteString(handle, "ATRMultTP=" + DoubleToString(g_atrMultiplierTP, 2) + "\n");
   FileWriteString(handle, "ATRTimeframe=" + IntegerToString((int)g_atrTimeframe) + "\n");
   FileWriteString(handle, "SpreadAdjSL=" + IntegerToString(g_spreadAdjSL) + "\n");
   FileWriteString(handle, "SpreadAdjTP=" + IntegerToString(g_spreadAdjTP) + "\n");
   FileWriteString(handle, "TPLocked=" + IntegerToString(g_tpLockedOnSL) + "\n");
   FileWriteString(handle, "TPMultiplier=" + DoubleToString(g_tpMultiplier, 2) + "\n");
   FileWriteString(handle, "MagicNumber=" + IntegerToString(g_magicNumber) + "\n");
   FileWriteString(handle, "Commentary=" + g_commentary + "\n");
   FileWriteString(handle, "TrailingStop=" + IntegerToString(g_trailingStop) + "\n");
   FileWriteString(handle, "BreakEven=" + IntegerToString(g_breakEven) + "\n");
   FileWriteString(handle, "CustomLeverage=" + DoubleToString(g_customLeverage, 2) + "\n");
   FileWriteString(handle, "DarkMode=" + IntegerToString(g_darkMode) + "\n");

   //--- Multi-TP
   FileWriteString(handle, "TPCount=" + IntegerToString(g_tpCount) + "\n");
   for(int i = 0; i < g_tpCount; i++)
   {
      FileWriteString(handle, "TP" + IntegerToString(i) + "Price=" + DoubleToString(g_tpLevels[i].price, 8) + "\n");
      FileWriteString(handle, "TP" + IntegerToString(i) + "Share=" + DoubleToString(g_tpLevels[i].sharePct, 2) + "\n");
   }

   FileClose(handle);
   return true;
}

//+------------------------------------------------------------------+
//| Load panel settings from file                                     |
//+------------------------------------------------------------------+
bool LoadSettings(string symbol = "")
{
   string filePath = GetSettingsFilePath(symbol);
   if(!FileIsExist(filePath)) return false;

   int handle = FileOpen(filePath, FILE_READ | FILE_TXT);
   if(handle == INVALID_HANDLE) return false;

   while(!FileIsEnding(handle))
   {
      string line = FileReadString(handle);
      if(StringLen(line) == 0) continue;

      int eqPos = StringFind(line, "=");
      if(eqPos < 0) continue;

      string key = StringSubstr(line, 0, eqPos);
      string val = StringSubstr(line, eqPos + 1);

      if(key == "PanelX")          g_panelState.posX = (int)StringToInteger(val);
      else if(key == "PanelY")     g_panelState.posY = (int)StringToInteger(val);
      else if(key == "Minimized")  g_panelState.isMinimized = (bool)StringToInteger(val);
      else if(key == "ActiveTab")  g_activeTab = (ENUM_PANEL_TAB)StringToInteger(val);
      else if(key == "LinesVisible") g_linesVisible = (bool)StringToInteger(val);
      else if(key == "TradeDir")   g_tradeDir = (ENUM_TRADE_DIR)StringToInteger(val);
      else if(key == "OrderMode")  g_orderMode = (ENUM_ORDER_MODE)StringToInteger(val);
      else if(key == "EntryPrice") g_entryPrice = StringToDouble(val);
      else if(key == "SLPrice")    g_stopLossPrice = StringToDouble(val);
      else if(key == "TPPrice")    g_takeProfitPrice = StringToDouble(val);
      else if(key == "StopPrice")  g_stopPrice = StringToDouble(val);
      else if(key == "SLInPoints") g_slInPoints = (bool)StringToInteger(val);
      else if(key == "TPInPoints") g_tpInPoints = (bool)StringToInteger(val);
      else if(key == "SLPointsVal")g_slPointsValue = (int)StringToInteger(val);
      else if(key == "TPPointsVal")g_tpPointsValue = (int)StringToInteger(val);
      else if(key == "RiskPercent")g_riskPercent = StringToDouble(val);
      else if(key == "RiskMoney")  g_riskMoney = StringToDouble(val);
      else if(key == "AccountBase")g_accountBase = (ENUM_ACCOUNT_BUTTON)StringToInteger(val);
      else if(key == "Commission") g_commission = StringToDouble(val);
      else if(key == "CommType")   g_commType = (ENUM_COMMISSION_TYPE)StringToInteger(val);
      else if(key == "ATRPeriod")  g_atrPeriod = (int)StringToInteger(val);
      else if(key == "ATRMultSL")  g_atrMultiplierSL = StringToDouble(val);
      else if(key == "ATRMultTP")  g_atrMultiplierTP = StringToDouble(val);
      else if(key == "ATRTimeframe") g_atrTimeframe = (ENUM_TIMEFRAMES)StringToInteger(val);
      else if(key == "SpreadAdjSL")g_spreadAdjSL = (bool)StringToInteger(val);
      else if(key == "SpreadAdjTP")g_spreadAdjTP = (bool)StringToInteger(val);
      else if(key == "TPLocked")   g_tpLockedOnSL = (bool)StringToInteger(val);
      else if(key == "TPMultiplier")g_tpMultiplier = StringToDouble(val);
      else if(key == "MagicNumber")g_magicNumber = StringToInteger(val);
      else if(key == "Commentary") g_commentary = val;
      else if(key == "TrailingStop")g_trailingStop = (int)StringToInteger(val);
      else if(key == "BreakEven")  g_breakEven = (int)StringToInteger(val);
      else if(key == "CustomLeverage") g_customLeverage = StringToDouble(val);
      else if(key == "DarkMode")   g_darkMode = (bool)StringToInteger(val);
      else if(key == "TPCount")
      {
         g_tpCount = (int)StringToInteger(val);
         if(g_tpCount < 1) g_tpCount = 1;
         ArrayResize(g_tpLevels, g_tpCount);
      }
      else
      {
         //--- Multi-TP parsing
         for(int i = 0; i < g_tpCount; i++)
         {
            if(key == "TP" + IntegerToString(i) + "Price")
               g_tpLevels[i].price = StringToDouble(val);
            else if(key == "TP" + IntegerToString(i) + "Share")
               g_tpLevels[i].sharePct = StringToDouble(val);
         }
      }
   }

   FileClose(handle);

   //--- Apply dark mode
   g_colors = g_darkMode ? GetDarkScheme() : GetLightScheme();
   g_panelState.activeTab = g_activeTab;
   g_panelState.linesVisible = g_linesVisible;

   return true;
}

//+------------------------------------------------------------------+
//| Delete settings file for a symbol                                 |
//+------------------------------------------------------------------+
bool DeleteSettings(string symbol = "")
{
   string filePath = GetSettingsFilePath(symbol);
   if(FileIsExist(filePath))
      return FileDelete(filePath);
   return true;
}

#endif // SETTINGS_MQH
