//+------------------------------------------------------------------+
//|                                                 TradeManager.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef TRADE_MANAGER_MQH
#define TRADE_MANAGER_MQH

#include <Trade\Trade.mqh>
#include "Defines.mqh"
#include "Globals.mqh"
#include "Utilities.mqh"
#include "Calculator.mqh"

//+------------------------------------------------------------------+
//| Trade safety checks — returns true if trade is allowed            |
//+------------------------------------------------------------------+
bool PreTradeChecks()
{
   string symbol = GetTradeSymbol();

   //--- Check if trading is disabled when lines hidden
   if(g_disableTradingHidden && !g_linesVisible)
   {
      Print("[BrokersDB PS] Trading disabled: lines are hidden");
      return false;
   }

   //--- Max spread check
   if(g_maxSpread > 0)
   {
      int spread = GetSpreadPoints(symbol);
      if(spread > g_maxSpread)
      {
         Print("[BrokersDB PS] Spread too wide: ", spread, " > ", g_maxSpread);
         return false;
      }
   }

   //--- Max Entry/SL distance
   if(g_maxEntrySLDist > 0)
   {
      int dist = CalcSLDistancePoints();
      if(dist > g_maxEntrySLDist)
      {
         Print("[BrokersDB PS] Entry/SL distance too large: ", dist, " > ", g_maxEntrySLDist);
         return false;
      }
   }

   //--- Min Entry/SL distance
   if(g_minEntrySLDist > 0)
   {
      int dist = CalcSLDistancePoints();
      if(dist < g_minEntrySLDist)
      {
         Print("[BrokersDB PS] Entry/SL distance too small: ", dist, " < ", g_minEntrySLDist);
         return false;
      }
   }

   //--- Max risk %
   if(g_maxRiskPct > 0)
   {
      if(g_calcResult.riskPercentResult > g_maxRiskPct)
      {
         Print("[BrokersDB PS] Risk too high: ", FormatPercent(g_calcResult.riskPercentResult), " > ", FormatPercent(g_maxRiskPct));
         return false;
      }
   }

   //--- Max trades total
   if(g_maxTradesTotal > 0)
   {
      int cnt = PositionsTotal() + OrdersTotal();
      if(cnt >= g_maxTradesTotal)
      {
         Print("[BrokersDB PS] Max trades reached: ", cnt, " >= ", g_maxTradesTotal);
         return false;
      }
   }

   //--- Max trades per symbol
   if(g_maxTradesSymbol > 0)
   {
      int cnt = CountTradesBySymbol(symbol);
      if(cnt >= g_maxTradesSymbol)
      {
         Print("[BrokersDB PS] Max trades per symbol reached: ", cnt);
         return false;
      }
   }

   //--- Max volume total
   if(g_maxVolTotal > 0)
   {
      double vol = GetTotalOpenVolume("");
      if(vol + g_calcResult.lotSizeAdjusted > g_maxVolTotal)
      {
         Print("[BrokersDB PS] Would exceed max total volume: ", vol + g_calcResult.lotSizeAdjusted);
         return false;
      }
   }

   //--- Max volume per symbol
   if(g_maxVolSymbol > 0)
   {
      double vol = GetTotalOpenVolume(symbol);
      if(vol + g_calcResult.lotSizeAdjusted > g_maxVolSymbol)
      {
         Print("[BrokersDB PS] Would exceed max symbol volume");
         return false;
      }
   }

   //--- Max total risk
   if(g_maxTotalRisk > 0)
   {
      CalculatePortfolioRisk();
      if(g_portfolioRisk.potentialRiskPercent > g_maxTotalRisk)
      {
         Print("[BrokersDB PS] Would exceed max total risk: ", FormatPercent(g_portfolioRisk.potentialRiskPercent));
         return false;
      }
   }

   //--- Max risk per symbol
   if(g_maxTotalRiskSym > 0)
   {
      // Simplified check using current position risk
      if(g_calcResult.riskPercentResult > g_maxTotalRiskSym)
      {
         Print("[BrokersDB PS] Would exceed max risk per symbol");
         return false;
      }
   }

   return true;
}

//+------------------------------------------------------------------+
//| Count trades (positions + orders) for a symbol                    |
//+------------------------------------------------------------------+
int CountTradesBySymbol(string symbol)
{
   int count = 0;
   for(int i = 0; i < PositionsTotal(); i++)
   {
      if(PositionGetSymbol(i) == symbol) count++;
   }
   for(int i = 0; i < OrdersTotal(); i++)
   {
      ulong ticket = OrderGetTicket(i);
      if(ticket > 0 && OrderGetString(ORDER_SYMBOL) == symbol) count++;
   }
   return count;
}

//+------------------------------------------------------------------+
//| Get total open volume (all or specific symbol)                    |
//+------------------------------------------------------------------+
double GetTotalOpenVolume(string symbol)
{
   double vol = 0;
   for(int i = 0; i < PositionsTotal(); i++)
   {
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0) continue;
      if(symbol == "" || PositionGetString(POSITION_SYMBOL) == symbol)
         vol += PositionGetDouble(POSITION_VOLUME);
   }
   for(int i = 0; i < OrdersTotal(); i++)
   {
      ulong ticket = OrderGetTicket(i);
      if(ticket == 0) continue;
      if(symbol == "" || OrderGetString(ORDER_SYMBOL) == symbol)
         vol += OrderGetDouble(ORDER_VOLUME_CURRENT);
   }
   return vol;
}

//+------------------------------------------------------------------+
//| Build order commentary with optional auto-suffix                  |
//+------------------------------------------------------------------+
string BuildCommentary()
{
   string comment = g_commentary;
   if(g_autoSuffix)
   {
      g_suffixCounter++;
      comment += "_" + IntegerToString(g_suffixCounter);
   }
   return comment;
}

//+------------------------------------------------------------------+
//| Execute a single trade                                            |
//+------------------------------------------------------------------+
bool ExecuteSingleTrade(double lots, double entry, double sl, double tp, string comment)
{
   string symbol = GetTradeSymbol();
   CTrade trade;
   trade.SetExpertMagicNumber(g_magicNumber);
   if(g_maxSlippage > 0)
      trade.SetDeviationInPoints(g_maxSlippage);
   trade.SetTypeFilling(ORDER_FILLING_FOK);

   //--- Apply SL/TP overrides
   double actualSL = g_noStopLoss ? 0 : sl;
   double actualTP = g_noTakeProfit ? 0 : tp;

   bool result = false;

   if(g_orderMode == ORDER_MODE_INSTANT)
   {
      //--- Market order
      if(g_tradeDir == TRADE_DIR_LONG)
         result = trade.Buy(lots, symbol, 0, actualSL, actualTP, comment);
      else
         result = trade.Sell(lots, symbol, 0, actualSL, actualTP, comment);
   }
   else if(g_orderMode == ORDER_MODE_PENDING)
   {
      //--- Pending order
      double currentPrice = GetCurrentPrice();
      datetime expiry = (g_expiryMinutes > 0) ? TimeCurrent() + g_expiryMinutes * 60 : 0;

      if(g_tradeDir == TRADE_DIR_LONG)
      {
         if(entry < currentPrice)
            result = trade.BuyLimit(lots, entry, symbol, actualSL, actualTP, ORDER_TIME_GTC, expiry, comment);
         else
            result = trade.BuyStop(lots, entry, symbol, actualSL, actualTP, ORDER_TIME_GTC, expiry, comment);
      }
      else
      {
         if(entry > currentPrice)
            result = trade.SellLimit(lots, entry, symbol, actualSL, actualTP, ORDER_TIME_GTC, expiry, comment);
         else
            result = trade.SellStop(lots, entry, symbol, actualSL, actualTP, ORDER_TIME_GTC, expiry, comment);
      }
   }
   else if(g_orderMode == ORDER_MODE_STOP_LIMIT)
   {
      //--- Stop Limit order (MT5 only)
      datetime expiry = (g_expiryMinutes > 0) ? TimeCurrent() + g_expiryMinutes * 60 : 0;
      if(g_tradeDir == TRADE_DIR_LONG)
         result = trade.OrderOpen(symbol, ORDER_TYPE_BUY_STOP_LIMIT, lots, entry, g_stopPrice, actualSL, actualTP, ORDER_TIME_GTC, expiry, comment);
      else
         result = trade.OrderOpen(symbol, ORDER_TYPE_SELL_STOP_LIMIT, lots, entry, g_stopPrice, actualSL, actualTP, ORDER_TIME_GTC, expiry, comment);
   }

   if(!result)
   {
      uint retcode = trade.ResultRetcode();
      Print("[BrokersDB PS] Trade failed: ", GetErrorDescription((int)retcode));
   }
   else if(!InpDisableTradeSounds)
   {
      PlaySound("ok.wav");
   }

   return result;
}

//+------------------------------------------------------------------+
//| Execute trade (single or multi-TP)                                |
//+------------------------------------------------------------------+
bool ExecuteTrade()
{
   if(!PreTradeChecks()) return false;

   //--- Ask for confirmation if enabled
   if(g_askConfirmation)
   {
      string opType = (g_tradeDir == TRADE_DIR_LONG) ? "Buy" : "Sell";
      string modeName = "Instant";
      if(g_orderMode == ORDER_MODE_PENDING) modeName = "Pending";
      if(g_orderMode == ORDER_MODE_STOP_LIMIT) modeName = "Stop Limit";
      
      string msg = "Do you want to execute this trade?\n\n";
      msg += "Type: " + opType + " " + modeName + "\n";
      msg += "Symbol: " + GetTradeSymbol() + "\n";
      msg += "Volume: " + DoubleToString(g_calcResult.lotSizeAdjusted, 2) + " lots\n";
      msg += "Risk: " + FormatMoneyShort(g_calcResult.riskMoneyResult) + " (" + FormatPercent(g_calcResult.riskPercentResult) + ")\n";
      
      if(MessageBox(msg, "BrokersDB Position Sizer - Confirm Trade", MB_YESNO | MB_ICONQUESTION) != IDYES)
      {
         Print("[BrokersDB PS] Trade cancelled by user.");
         return false;
      }
   }

   double entry = GetEntryPrice();
   double sl = g_stopLossPrice;
   string comment = BuildCommentary();
   double lots = g_calcResult.lotSizeAdjusted;

   //--- Subtract open/pending volume
   if(g_subtractOPV)
   {
      double openVol = GetTotalOpenVolume(GetTradeSymbol());
      lots -= openVol;
      if(lots <= 0) { Print("[BrokersDB PS] No volume to trade after OPV subtract"); return false; }
      lots = NormalizeLots(lots);
   }
   if(g_subtractPOV)
   {
      double pendVol = 0;
      string sym = GetTradeSymbol();
      for(int i = 0; i < OrdersTotal(); i++)
      {
         ulong ticket = OrderGetTicket(i);
         if(ticket > 0 && OrderGetString(ORDER_SYMBOL) == sym)
            pendVol += OrderGetDouble(ORDER_VOLUME_CURRENT);
      }
      lots -= pendVol;
      if(lots <= 0) { Print("[BrokersDB PS] No volume after POV subtract"); return false; }
      lots = NormalizeLots(lots);
   }

   //--- Multi-TP trades
   if(g_tpCount > 1)
   {
      bool allOk = true;
      for(int i = 0; i < g_tpCount; i++)
      {
         double tpLots = g_tpLevels[i].lots;
         double tp = g_tpLevels[i].price;
         if(tpLots <= 0) continue;

         if(!ExecuteSingleTrade(tpLots, entry, sl, tp, comment))
            allOk = false;
      }
      return allOk;
   }

   //--- Single trade
   return ExecuteSingleTrade(lots, entry, sl, g_takeProfitPrice, comment);
}

//+------------------------------------------------------------------+
//| Apply trailing stop to positions with matching magic number       |
//+------------------------------------------------------------------+
void ApplyTrailingStop()
{
   if(g_trailingStop <= 0) return;

   string symbol = GetTradeSymbol();
   double trailDist = PointsToPrice(g_trailingStop, symbol);

   for(int i = PositionsTotal() - 1; i >= 0; i--)
   {
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0) continue;
      if(!PositionSelectByTicket(ticket)) continue;
      if(PositionGetInteger(POSITION_MAGIC) != g_magicNumber) continue;
      if(PositionGetString(POSITION_SYMBOL) != symbol) continue;

      ENUM_POSITION_TYPE type = (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE);
      double currentSL = PositionGetDouble(POSITION_SL);
      double currentTP = PositionGetDouble(POSITION_TP);

      CTrade trade;
      trade.SetExpertMagicNumber(g_magicNumber);

      if(type == POSITION_TYPE_BUY)
      {
         double bid = GetBid(symbol);
         double newSL = NormalizePrice(bid - trailDist, symbol);
         if(newSL > currentSL || currentSL == 0)
            trade.PositionModify(ticket, newSL, currentTP);
      }
      else
      {
         double ask = GetAsk(symbol);
         double newSL = NormalizePrice(ask + trailDist, symbol);
         if(newSL < currentSL || currentSL == 0)
            trade.PositionModify(ticket, newSL, currentTP);
      }
   }
}

//+------------------------------------------------------------------+
//| Apply breakeven to positions with matching magic number           |
//+------------------------------------------------------------------+
void ApplyBreakEven()
{
   if(g_breakEven <= 0) return;

   string symbol = GetTradeSymbol();
   double beDist = PointsToPrice(g_breakEven, symbol);

   for(int i = PositionsTotal() - 1; i >= 0; i--)
   {
      ulong ticket = PositionGetTicket(i);
      if(ticket == 0) continue;
      if(!PositionSelectByTicket(ticket)) continue;
      if(PositionGetInteger(POSITION_MAGIC) != g_magicNumber) continue;
      if(PositionGetString(POSITION_SYMBOL) != symbol) continue;

      ENUM_POSITION_TYPE type = (ENUM_POSITION_TYPE)PositionGetInteger(POSITION_TYPE);
      double openPrice = PositionGetDouble(POSITION_PRICE_OPEN);
      double currentSL = PositionGetDouble(POSITION_SL);
      double currentTP = PositionGetDouble(POSITION_TP);

      CTrade trade;
      trade.SetExpertMagicNumber(g_magicNumber);

      if(type == POSITION_TYPE_BUY)
      {
         double bid = GetBid(symbol);
         if(bid >= openPrice + beDist && (currentSL < openPrice || currentSL == 0))
            trade.PositionModify(ticket, openPrice, currentTP);
      }
      else
      {
         double ask = GetAsk(symbol);
         if(ask <= openPrice - beDist && (currentSL > openPrice || currentSL == 0))
            trade.PositionModify(ticket, openPrice, currentTP);
      }
   }
}

#endif // TRADE_MANAGER_MQH
