//+------------------------------------------------------------------+
//|                                                    Utilities.mqh |
//|                        Copyright 2025-2026, BrokersDB.com        |
//|                                     https://www.brokersdb.com    |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025-2026, BrokersDB.com"
#property link      "https://www.brokersdb.com"
#property strict

#ifndef UTILITIES_MQH
#define UTILITIES_MQH

#include "Defines.mqh"
#include "Globals.mqh"

//+------------------------------------------------------------------+
//| Get the proper symbol for trading                                 |
//+------------------------------------------------------------------+
string GetTradeSymbol()
{
   if(g_tradeSymbol != "" && g_tradeSymbol != _Symbol)
      return g_tradeSymbol;
   return _Symbol;
}

//+------------------------------------------------------------------+
//| Get account size based on selected mode                           |
//+------------------------------------------------------------------+
double GetAccountSize()
{
   double size = 0;

   //--- Custom balance overrides everything
   if(InpCustomBalance > 0)
      return InpCustomBalance;

   switch(g_accountBase)
   {
      case ACCT_BALANCE:
         size = AccountInfoDouble(ACCOUNT_BALANCE);
         break;
      case ACCT_EQUITY:
         size = AccountInfoDouble(ACCOUNT_EQUITY);
         break;
      case ACCT_BALANCE_MINUS_CPR:
         size = AccountInfoDouble(ACCOUNT_BALANCE) - g_portfolioRisk.currentRiskMoney;
         break;
   }

   //--- Add additional funds if set
   if(InpAdditionalFunds != 0 && InpCustomBalance == 0)
      size += InpAdditionalFunds;

   return size;
}

//+------------------------------------------------------------------+
//| Format a double to string with proper digits                      |
//+------------------------------------------------------------------+
string FormatDouble(double value, int digits)
{
   return DoubleToString(value, digits);
}

//+------------------------------------------------------------------+
//| Format money value with 2 decimal places                          |
//+------------------------------------------------------------------+
string FormatMoney(double value)
{
   string currency = AccountInfoString(ACCOUNT_CURRENCY);
   if(MathAbs(value) >= 1000000)
      return FormatDouble(value, 0) + " " + currency;
   return FormatDouble(value, 2) + " " + currency;
}

//+------------------------------------------------------------------+
//| Format money value without currency suffix                        |
//+------------------------------------------------------------------+
string FormatMoneyShort(double value)
{
   if(MathAbs(value) >= 1000000)
      return FormatDouble(value, 0);
   return FormatDouble(value, 2);
}

//+------------------------------------------------------------------+
//| Format percentage value                                           |
//+------------------------------------------------------------------+
string FormatPercent(double value)
{
   return FormatDouble(value, 2) + "%";
}

//+------------------------------------------------------------------+
//| Get spread in points                                              |
//+------------------------------------------------------------------+
int GetSpreadPoints(string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   return (int)SymbolInfoInteger(symbol, SYMBOL_SPREAD);
}

//+------------------------------------------------------------------+
//| Get point value (tick value normalized to 1 point)                |
//+------------------------------------------------------------------+
double GetPointValue(string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   double tickSize  = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_SIZE);
   double tickValue = SymbolInfoDouble(symbol, SYMBOL_TRADE_TICK_VALUE);
   double point     = SymbolInfoDouble(symbol, SYMBOL_POINT);

   if(tickSize == 0) return 0;
   return tickValue * point / tickSize;
}

//+------------------------------------------------------------------+
//| Convert points to price distance                                  |
//+------------------------------------------------------------------+
double PointsToPrice(int points, string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   return points * SymbolInfoDouble(symbol, SYMBOL_POINT);
}

//+------------------------------------------------------------------+
//| Convert price distance to points                                  |
//+------------------------------------------------------------------+
int PriceToPoints(double priceDistance, string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   double point = SymbolInfoDouble(symbol, SYMBOL_POINT);
   if(point == 0) return 0;
   return (int)MathRound(priceDistance / point);
}

//+------------------------------------------------------------------+
//| Get number of decimal digits for symbol                           |
//+------------------------------------------------------------------+
int GetSymbolDigits(string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   return (int)SymbolInfoInteger(symbol, SYMBOL_DIGITS);
}

//+------------------------------------------------------------------+
//| Normalize price to symbol digits                                  |
//+------------------------------------------------------------------+
double NormalizePrice(double price, string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   int digits = GetSymbolDigits(symbol);
   return NormalizeDouble(price, digits);
}

//+------------------------------------------------------------------+
//| Normalize lot size according to broker restrictions                |
//+------------------------------------------------------------------+
double NormalizeLots(double lots, string symbol = "")
{
   if(symbol == "") symbol = _Symbol;

   double minLot  = SymbolInfoDouble(symbol, SYMBOL_VOLUME_MIN);
   double maxLot  = SymbolInfoDouble(symbol, SYMBOL_VOLUME_MAX);
   double lotStep = SymbolInfoDouble(symbol, SYMBOL_VOLUME_STEP);

   if(lotStep == 0) lotStep = 0.01;

   //--- Round to lot step
   if(InpRoundDown)
      lots = MathFloor(lots / lotStep) * lotStep;
   else
      lots = MathRound(lots / lotStep) * lotStep;

   //--- Clamp to min/max
   if(!InpUnadjustedPS)
   {
      if(lots < minLot) lots = minLot;
      if(lots > maxLot) lots = maxLot;
   }

   //--- Normalize to remove floating point artifacts
   int lotDigits = 0;
   double tmpStep = lotStep;
   while(MathFloor(tmpStep) != tmpStep && lotDigits < 10)
   {
      tmpStep *= 10;
      lotDigits++;
   }

   return NormalizeDouble(lots, lotDigits);
}

//+------------------------------------------------------------------+
//| Create prefixed object name                                       |
//+------------------------------------------------------------------+
string ObjName(string baseName)
{
   return g_prefix + baseName;
}

//+------------------------------------------------------------------+
//| Safely create a rectangle label (panel background element)        |
//+------------------------------------------------------------------+
bool CreateRectLabel(long chart, string name, int x, int y, int width, int height,
                     color bgColor, color borderColor, int borderWidth = 1,
                     ENUM_BASE_CORNER corner = CORNER_LEFT_UPPER)
{
   if(ObjectFind(chart, name) >= 0)
      ObjectDelete(chart, name);

   if(!ObjectCreate(chart, name, OBJ_RECTANGLE_LABEL, 0, 0, 0))
      return false;

   ObjectSetInteger(chart, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(chart, name, OBJPROP_YDISTANCE, y);
   ObjectSetInteger(chart, name, OBJPROP_XSIZE, width);
   ObjectSetInteger(chart, name, OBJPROP_YSIZE, height);
   ObjectSetInteger(chart, name, OBJPROP_BGCOLOR, bgColor);
   ObjectSetInteger(chart, name, OBJPROP_BORDER_COLOR, borderColor);
   ObjectSetInteger(chart, name, OBJPROP_BORDER_TYPE, BORDER_FLAT);
   ObjectSetInteger(chart, name, OBJPROP_WIDTH, borderWidth);
   ObjectSetInteger(chart, name, OBJPROP_CORNER, corner);
   ObjectSetInteger(chart, name, OBJPROP_BACK, false);
   ObjectSetInteger(chart, name, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(chart, name, OBJPROP_HIDDEN, true);
   ObjectSetInteger(chart, name, OBJPROP_ZORDER, 0);

   return true;
}

//+------------------------------------------------------------------+
//| Safely create a text label                                        |
//+------------------------------------------------------------------+
bool CreateLabel(long chart, string name, int x, int y, string text,
                 color textColor, int fontSize = 0, string fontName = "",
                 ENUM_BASE_CORNER corner = CORNER_LEFT_UPPER,
                 ENUM_ANCHOR_POINT anchor = ANCHOR_LEFT_UPPER)
{
   if(fontSize == 0) fontSize = PANEL_FONT_SIZE;
   if(fontName == "") fontName = PANEL_FONT_NAME;

   if(ObjectFind(chart, name) >= 0)
      ObjectDelete(chart, name);

   if(!ObjectCreate(chart, name, OBJ_LABEL, 0, 0, 0))
      return false;

   ObjectSetInteger(chart, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(chart, name, OBJPROP_YDISTANCE, y);
   ObjectSetInteger(chart, name, OBJPROP_CORNER, corner);
   ObjectSetInteger(chart, name, OBJPROP_ANCHOR, anchor);
   ObjectSetString(chart, name, OBJPROP_TEXT, text);
   ObjectSetInteger(chart, name, OBJPROP_COLOR, textColor);
   ObjectSetInteger(chart, name, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(chart, name, OBJPROP_FONT, fontName);
   ObjectSetInteger(chart, name, OBJPROP_BACK, false);
   ObjectSetInteger(chart, name, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(chart, name, OBJPROP_HIDDEN, true);
   ObjectSetInteger(chart, name, OBJPROP_ZORDER, 5);

   return true;
}

//+------------------------------------------------------------------+
//| Safely create an edit box                                         |
//+------------------------------------------------------------------+
bool CreateEdit(long chart, string name, int x, int y, int width, int height,
                string text, color textColor, color bgColor, color borderColor,
                int fontSize = 0, string fontName = "",
                ENUM_BASE_CORNER corner = CORNER_LEFT_UPPER,
                ENUM_ALIGN_MODE align = ALIGN_RIGHT,
                bool readOnly = false)
{
   if(fontSize == 0) fontSize = PANEL_FONT_SIZE;
   if(fontName == "") fontName = PANEL_FONT_NAME;

   if(ObjectFind(chart, name) >= 0)
      ObjectDelete(chart, name);

   if(!ObjectCreate(chart, name, OBJ_EDIT, 0, 0, 0))
      return false;

   ObjectSetInteger(chart, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(chart, name, OBJPROP_YDISTANCE, y);
   ObjectSetInteger(chart, name, OBJPROP_XSIZE, width);
   ObjectSetInteger(chart, name, OBJPROP_YSIZE, height);
   ObjectSetString(chart, name, OBJPROP_TEXT, text);
   ObjectSetInteger(chart, name, OBJPROP_COLOR, textColor);
   ObjectSetInteger(chart, name, OBJPROP_BGCOLOR, bgColor);
   ObjectSetInteger(chart, name, OBJPROP_BORDER_COLOR, borderColor);
   ObjectSetInteger(chart, name, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(chart, name, OBJPROP_FONT, fontName);
   ObjectSetInteger(chart, name, OBJPROP_CORNER, corner);
   ObjectSetInteger(chart, name, OBJPROP_ALIGN, align);
   ObjectSetInteger(chart, name, OBJPROP_READONLY, readOnly);
   ObjectSetInteger(chart, name, OBJPROP_BACK, false);
   ObjectSetInteger(chart, name, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(chart, name, OBJPROP_HIDDEN, true);
   ObjectSetInteger(chart, name, OBJPROP_ZORDER, 10);

   return true;
}

//+------------------------------------------------------------------+
//| Safely create a button                                            |
//+------------------------------------------------------------------+
bool CreateButton(long chart, string name, int x, int y, int width, int height,
                  string text, color textColor, color bgColor,
                  int fontSize = 0, string fontName = "",
                  ENUM_BASE_CORNER corner = CORNER_LEFT_UPPER)
{
   if(fontSize == 0) fontSize = PANEL_FONT_SIZE;
   if(fontName == "") fontName = PANEL_FONT_NAME;

   if(ObjectFind(chart, name) >= 0)
      ObjectDelete(chart, name);

   if(!ObjectCreate(chart, name, OBJ_BUTTON, 0, 0, 0))
      return false;

   ObjectSetInteger(chart, name, OBJPROP_XDISTANCE, x);
   ObjectSetInteger(chart, name, OBJPROP_YDISTANCE, y);
   ObjectSetInteger(chart, name, OBJPROP_XSIZE, width);
   ObjectSetInteger(chart, name, OBJPROP_YSIZE, height);
   ObjectSetString(chart, name, OBJPROP_TEXT, text);
   ObjectSetInteger(chart, name, OBJPROP_COLOR, textColor);
   ObjectSetInteger(chart, name, OBJPROP_BGCOLOR, bgColor);
   ObjectSetInteger(chart, name, OBJPROP_BORDER_COLOR, bgColor);
   ObjectSetInteger(chart, name, OBJPROP_FONTSIZE, fontSize);
   ObjectSetString(chart, name, OBJPROP_FONT, fontName);
   ObjectSetInteger(chart, name, OBJPROP_CORNER, corner);
   ObjectSetInteger(chart, name, OBJPROP_BACK, false);
   ObjectSetInteger(chart, name, OBJPROP_SELECTABLE, false);
   ObjectSetInteger(chart, name, OBJPROP_HIDDEN, true);
   ObjectSetInteger(chart, name, OBJPROP_STATE, false);
   ObjectSetInteger(chart, name, OBJPROP_ZORDER, 10);

   return true;
}

//+------------------------------------------------------------------+
//| Delete all objects with our prefix                                |
//+------------------------------------------------------------------+
void DeleteAllObjects(long chart = 0)
{
   int total = ObjectsTotal(chart, 0, -1);
   for(int i = total - 1; i >= 0; i--)
   {
      string name = ObjectName(chart, i, 0, -1);
      if(StringFind(name, g_prefix) == 0)
         ObjectDelete(chart, name);
   }
}

//+------------------------------------------------------------------+
//| Check if an object was clicked by name                            |
//+------------------------------------------------------------------+
bool IsObjectClicked(string clickedName, string baseName)
{
   return (clickedName == ObjName(baseName));
}

//+------------------------------------------------------------------+
//| Get the Ask price for the trade symbol                            |
//+------------------------------------------------------------------+
double GetAsk(string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   return SymbolInfoDouble(symbol, SYMBOL_ASK);
}

//+------------------------------------------------------------------+
//| Get the Bid price for the trade symbol                            |
//+------------------------------------------------------------------+
double GetBid(string symbol = "")
{
   if(symbol == "") symbol = _Symbol;
   return SymbolInfoDouble(symbol, SYMBOL_BID);
}

//+------------------------------------------------------------------+
//| Get current price based on trade direction                        |
//+------------------------------------------------------------------+
double GetCurrentPrice()
{
   if(g_tradeDir == TRADE_DIR_LONG)
      return GetAsk();
   return GetBid();
}

//+------------------------------------------------------------------+
//| Get the entry price (current for instant, user-set for pending)   |
//+------------------------------------------------------------------+
double GetEntryPrice()
{
   if(g_orderMode == ORDER_MODE_INSTANT)
      return GetCurrentPrice();
   return g_entryPrice;
}

//+------------------------------------------------------------------+
//| Parse hotkey string (e.g. "Shift+T") into key code and modifiers  |
//+------------------------------------------------------------------+
struct SHotkey
{
   int   keyCode;
   bool  shift;
   bool  ctrl;
};

SHotkey ParseHotkey(string hkString)
{
   SHotkey hk;
   hk.keyCode = 0;
   hk.shift = false;
   hk.ctrl = false;

   StringToUpper(hkString);
   StringTrimLeft(hkString);
   StringTrimRight(hkString);

   //--- Check modifiers
   if(StringFind(hkString, "SHIFT+") >= 0)
   {
      hk.shift = true;
      StringReplace(hkString, "SHIFT+", "");
   }
   if(StringFind(hkString, "CTRL+") >= 0)
   {
      hk.ctrl = true;
      StringReplace(hkString, "CTRL+", "");
   }

   StringTrimLeft(hkString);
   StringTrimRight(hkString);

   //--- Map key name to code
   if(hkString == "TAB")         hk.keyCode = 9;
   else if(hkString == "ENTER")  hk.keyCode = 13;
   else if(hkString == "ESCAPE") hk.keyCode = 27;
   else if(hkString == "SPACE")  hk.keyCode = 32;
   else if(hkString == "`")      hk.keyCode = 192;
   else if(StringLen(hkString) == 1)
   {
      ushort ch = StringGetCharacter(hkString, 0);
      hk.keyCode = (int)ch;
   }

   return hk;
}

//+------------------------------------------------------------------+
//| Get error description from error code                             |
//+------------------------------------------------------------------+
string GetErrorDescription(int errorCode)
{
   switch(errorCode)
   {
      case 10004: return "Requote";
      case 10006: return "Request rejected";
      case 10007: return "Request canceled by trader";
      case 10008: return "Order placed";
      case 10009: return "Request completed";
      case 10010: return "Only part of the request was completed";
      case 10011: return "Request processing error";
      case 10012: return "Request canceled by timeout";
      case 10013: return "Invalid request";
      case 10014: return "Invalid volume in the request";
      case 10015: return "Invalid price in the request";
      case 10016: return "Invalid stops in the request";
      case 10017: return "Trade is disabled";
      case 10018: return "Market is closed";
      case 10019: return "There is not enough money to complete the request";
      case 10020: return "Prices changed";
      case 10021: return "There are no quotes to process the request";
      case 10022: return "Invalid order expiration date in the request";
      case 10023: return "Order state changed";
      case 10024: return "Too frequent requests";
      case 10025: return "No changes in request";
      case 10026: return "Autotrading disabled by server";
      case 10027: return "Autotrading disabled by client terminal";
      case 10028: return "Request locked for processing";
      case 10029: return "Order or position frozen";
      case 10030: return "Invalid order filling type";
      case 10031: return "No connection with the trade server";
      case 10032: return "Operation only for live accounts";
      case 10033: return "Pending orders limit reached";
      case 10034: return "Volume limit for symbol reached";
      case 10035: return "Incorrect or prohibited order type";
      case 10036: return "Position with specified POSITION_IDENTIFIER has already been closed";
      case 10038: return "Close volume exceeds the current position volume";
      case 10039: return "Close order already exists for specified position";
      case 10040: return "Maximum number of open positions reached";
      case 10041: return "Pending order activation request rejected, order canceled";
      case 10042: return "Request rejected, only long positions are allowed";
      case 10043: return "Request rejected, only short positions are allowed";
      case 10044: return "Request rejected, only position close operations are allowed";
      default:    return "Unknown error (" + IntegerToString(errorCode) + ")";
   }
}

#endif // UTILITIES_MQH
